// Battery monitor.
//
// Units:      ADC
// Interrupts: none
// Pins:       ADC1 (PORTC1)
// Clock:      8Mhz
//

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

#define ADC_MAX_COUNT 0x3ff

// Take a reading on currently selected adc.
// Taken:    nothing
// Returned: adc counts (0..ADC_MAX_COUNT)
//
static WORD
ADC_read()
   {
   // start adc converter
   ADCSRA |= (1 << ADSC);

   // wait for conversion to complete
   while (ADCSRA & (1 << ADSC));

   // read out the result
   BYTE lsb = ADCL;
   BYTE msb = ADCH;
   WORD val = (msb << 8) | lsb;
   return val;
   }

// Take a reading on ADC1 (external battery monitor).
// Taken:    nothing
// Returned: adc counts (0..ADC_MAX_COUNT)
//
static WORD
ADC_read_battery()
   {
   ADMUX  = 0             
          | (1 << REFS0)  // voltage reference = internal 1.1v
          | (1 << REFS1)

          | (1 << MUX0)   // voltage source = ADC1
          | (0 << MUX1)
          | (0 << MUX2)
          | (0 << MUX3)
          ;
   return ADC_read();
   }

// Convert battery adc reading to volts.
// Returned: battery reading in millivolts
//
static MILLIVOLTS
ADC_battery_counts_to_millivolts(WORD adc_counts)
   {
   return (WORD)( adc_counts * 10000UL / Config.battery_divisor );
   }

// --------------------------------------------------------------------
//                          Interface.
// --------------------------------------------------------------------

// Prepare adc for use.
//
static void
BAT_init()
   {
   // adc needs a 50-200kHz clock
   // lower clock  = more accurate
   // higher clock = faster conversion
   // for system clock of 8Mhz:
   //    prescaler  adc clock
   //       128      62.5kHz
   //       64       125 kHz
   //       32       250 kHz

   #if CLOCK_MHZ != 8
   #error clock!
   #endif
   
   ADCSRA = 0
          | (1 << ADPS0)  // adc prescaler = divide by 128
          | (1 << ADPS1)
          | (1 << ADPS2)
          
          | (1 << ADEN)   // enable adc
          ;

   // run adc once to put it in a clean state
   ADC_read(); 
   }

// Battery reading, in millivolts.
//
static MILLIVOLTS
BAT_read()
   {
   return ADC_battery_counts_to_millivolts(ADC_read_battery());
   }
