// Save and recall configuration constants.
//

// Configuration constants stored in eeprom.
//
typedef struct 
   {
   BYTE       gyro_twi_address;                // 0x69 for L3G-4200D (old chip), 0x6b for L3G-D20 (new chip)
   SWORD      gyro_x_tc, gyro_y_tc, gyro_z_tc; // zero rate bias temperature coefficients (typically 0 to +/- 50)
   GYRO_RATE  gyro_comp;                       // divisor to compensate for directional asymmetry in roll axis rate sensitivity (300-500)
   GYRO_ANGLE gyro_sensitivity;                // gyro-digits to tenths-of-a-degree divisor (datasheet=1143)

   SWORD      servo_bias;                      // servo centering bias in tenths-of-a-degree (typically 0 to +/-30)

   DWORD      battery_divisor;                 // battery adc-digits to millivolts divisor (~990)

   SDWORD     camera_gear_pos;                 // camera/servo drive ratio in positive direction (~150)
   SDWORD     camera_gear_neg;                 // camera/servo drive ratio in negative direction (~150)
   
   DWORD      checksum;                        // sum of the above
   } CONFIG;

// Ram copy of configuration constants.
//
CONFIG Config;

// Compute checksum of ram copy.
//
static DWORD
CONFIG_checksum()
   {
   DWORD checksum = 0;
   checksum += Config.gyro_twi_address;
   checksum += Config.gyro_x_tc;
   checksum += Config.gyro_y_tc;
   checksum += Config.gyro_z_tc;
   checksum += Config.gyro_comp;
   checksum += Config.gyro_sensitivity;
   checksum += Config.servo_bias;
   checksum += Config.battery_divisor;
   checksum += Config.camera_gear_pos;
   checksum += Config.camera_gear_neg;
   return checksum;
   }


// Initialize ram copy.
// Note: the "tc" temperature compensation constants are obtained using "./tc/plot" on data generated by "adjust_camera()"
// Example:
//   ------ curve fitting --------   ------- constants ----
//   x-bias = ( 89.1) + (-5.1) * t   Config.gyro_x_tc = -51;
//   y-bias = ( 16.2) + ( 0.1) * t   Config.gyro_y_tc =   1;
//   z-bias = (-99.2) + ( 2.0) * t   Config.gyro_z_tc =  20;
//
static void
CONFIG_init()
   {

#if 0 // template

   Config.gyro_twi_address   = 0x69; // or 0x6b
   Config.gyro_x_tc          = 0;
   Config.gyro_y_tc          = 0;
   Config.gyro_z_tc          = 0;
   Config.gyro_comp          = 1000;
   Config.gyro_sensitivity   = 1143;
   Config.servo_bias         = 0;
   Config.battery_divisor    = 1000;
   Config.camera_gear_pos    = 100;
   Config.camera_gear_neg    = 100;
   
#elif 0 // camera 2

   Config.gyro_twi_address   = 0x69;
   Config.gyro_x_tc          = -51;
   Config.gyro_y_tc          =   2;
   Config.gyro_z_tc          =  15;
   Config.gyro_comp          = 500;
   Config.gyro_sensitivity   = 1187;
   Config.servo_bias         = 30;
   Config.battery_divisor    = 976;
   Config.camera_gear_pos    = 155;
   Config.camera_gear_neg    = 150;

#elif 0 // camera 3

   Config.gyro_twi_address   = 0x6b;
   Config.gyro_x_tc          =  21;
   Config.gyro_y_tc          =  27;
   Config.gyro_z_tc          = 101;
   Config.gyro_comp          = -300;
   Config.gyro_sensitivity   = 1075;
   Config.servo_bias         = -60;
   Config.battery_divisor    = 1014;
   Config.camera_gear_pos    = 154;
   Config.camera_gear_neg    = 150;

#elif 0 // camera 4

   Config.gyro_twi_address   = 0x69;
   Config.gyro_x_tc          = -32;
   Config.gyro_y_tc          =  -8;
   Config.gyro_z_tc          =   1;
   Config.gyro_comp          = 1000;
   Config.gyro_sensitivity   = 1214;
   Config.servo_bias         =   0;
   Config.battery_divisor    = 986;
   Config.camera_gear_pos    = 160;
   Config.camera_gear_neg    = 150;

#else
   
   printf("noconfig\n");

#endif
   }

// Load ram copy from eeprom.
//
static void
CONFIG_load()
   {
   EEPROM_read_block(0, &Config, sizeof(Config));
   if (Config.checksum != CONFIG_checksum())
      {
      printf("new camera\n");
      CONFIG_init();
      }
   }

// Save ram copy to eeprom.
//
static void
CONFIG_save()
   {
   Config.checksum = CONFIG_checksum();
   EEPROM_write_block(0, &Config, sizeof(Config));
   printf("config saved\n");
   }

// Display current ram copy.
//
void
CONFIG_dump()
   {
   printf("%02x ",       Config.gyro_twi_address);
   printf("(%d %d %d) ", Config.gyro_x_tc, Config.gyro_y_tc, Config.gyro_z_tc);
   printf("%ld ",        Config.gyro_comp);
   printf("%ld) ",       Config.gyro_sensitivity);
   printf("%d ",         Config.servo_bias);
   printf("%lu ",        Config.battery_divisor);
   printf("(%ld ",       Config.camera_gear_pos);
   printf("%ld) ",       Config.camera_gear_neg);
   printf("0x%lx\n",     Config.checksum);
   }
