// Controller for STmicroelectronics L3G-4200D and L3G-D20 three axis gyros.
//
// Units:      TIMER0
// Interrupts: TIMER0
// Pins:       SDA,SCL
// Clock:      8Mhz
//
// Gyro characteristics:
//
//    range        +/- 250 degrees-per-second
//    resolution   +/- 15 bits
//    sensitivity  .00875 degrees-per-second per gyro-digit
//    update rate  100/200/400/800 Hz for L3G-4200D
//                  95/190/380/760 Hz for L3G-D20  
//
// Note: the -D20 supposedly has a higher resonant frequency for better noise and vibration immunity in "acoustical" region.
//

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

// Gyro chip's register addresses.
//
#define L3G_WHO_AM_I      0x0F

#define L3G_CTRL_REG1     0x20
#define L3G_CTRL_REG2     0x21
#define L3G_CTRL_REG3     0x22
#define L3G_CTRL_REG4     0x23
#define L3G_CTRL_REG5     0x24
#define L3G_REFERENCE     0x25
#define L3G_OUT_TEMP      0x26
#define L3G_STATUS_REG    0x27

#define L3G_OUT_X_L       0x28
#define L3G_OUT_X_H       0x29
#define L3G_OUT_Y_L       0x2A
#define L3G_OUT_Y_H       0x2B
#define L3G_OUT_Z_L       0x2C
#define L3G_OUT_Z_H       0x2D

#define L3G_FIFO_CTRL_REG 0x2E
#define L3G_FIFO_SRC_REG  0x2F

#define L3G_INT1_CFG      0x30
#define L3G_INT1_SRC      0x31
#define L3G_INT1_THS_XH   0x32
#define L3G_INT1_THS_XL   0x33
#define L3G_INT1_THS_YH   0x34
#define L3G_INT1_THS_YL   0x35
#define L3G_INT1_THS_ZH   0x36
#define L3G_INT1_THS_ZL   0x37
#define L3G_INT1_DURATION 0x38

// Gyro chip's TWI address, as packaged on Pololu carrier board.
//   110.1001 = 0x69 for L3G-4200D (old chip)
//   110.1011 = 0x6b for L3G-D20   (new chip)
//
#define GYRO_ADDRESS Config.gyro_twi_address

#define L3G_I_AM_4200D 0xD3 // response to L3G_WHO_AM_I for L3G-4200D (old chip)
#define L3G_I_AM_D20   0xD4 // response to L3G_WHO_AM_I for L3G-D20   (new chip)

// Read gyro sensors.
//
static void
GYRO_read(GYRO_RATE *xp, GYRO_RATE *yp, GYRO_RATE *zp)
   {
   BYTE b[6];
   TWI_read_multi(GYRO_ADDRESS, L3G_OUT_X_L | 0x80, sizeof(b), b); // 0x80 is autoincrement bit
   
   *xp = b[0] | (b[1] << 8);
   *yp = b[2] | (b[3] << 8);
   *zp = b[4] | (b[5] << 8);
   }

// Temperature compensation data.
//
static          GYRO_RATE  GYRO_x_bias_ref; // reference zero-rate bias, set by GYRO_calibrate()
static          GYRO_RATE  GYRO_y_bias_ref; // "
static          GYRO_RATE  GYRO_z_bias_ref; // "
static          SBYTE      GYRO_t_ref;      // die temperature for which the above biases are valid

// Divisor to compensate for directional asymmetry in roll axis rate sensitivity.
//
#define GYRO_comp Config.gyro_comp         

// Interrupt communication area.
//
static volatile GYRO_TICKS GYRO_ticks;   // elapsed time so far (~200Hz)
static volatile DWORD      GYRO_cnt;     // number of measurements taken so far (~100Hz)
static volatile WORD       GYRO_misses;  // number of times gyro data changed before we could read it (should only happen once, at startup)

static volatile GYRO_RATE  GYRO_x_bias;  // current zero-rate bias, temperature corrected every few seconds
static volatile GYRO_RATE  GYRO_y_bias;  // "
static volatile GYRO_RATE  GYRO_z_bias;  // "

static volatile GYRO_RATE  GYRO_x_rate;  // current gyro rate, updated at 100Hz
static volatile GYRO_RATE  GYRO_y_rate;  // "
static volatile GYRO_RATE  GYRO_z_rate;  // "

static volatile GYRO_ANGLE GYRO_x_angle; // current gyro angle, integrated from rate measurements
static volatile GYRO_ANGLE GYRO_y_angle; // "
static volatile GYRO_ANGLE GYRO_z_angle; // "

// Handler for timer tick interrupt.
// We use this interrupt to integrate angular rate measurements, yielding an angle.
// Note: not all of these values are used - we can omit them to make more cpu cycles available
//       outside the interrupt routine, in the mainline code.
//
static void
GYRO_interrupt()
   {
   BYTE status = TWI_read(GYRO_ADDRESS, L3G_STATUS_REG);

   if (status & 0x08)
      { // a new set of data is available (note: interrupt fires at ~200Hz but gyro data changes at ~100Hz)
      
      GYRO_RATE x_raw, y_raw, z_raw;
      GYRO_read(&x_raw, &y_raw, &z_raw);
      
      GYRO_RATE x = x_raw - GYRO_x_bias;
      GYRO_RATE y = y_raw - GYRO_y_bias;
      GYRO_RATE z = z_raw - GYRO_z_bias;

      // integrate rates to yield angles
      //
      GYRO_x_angle += x;
      GYRO_y_angle += y;
      GYRO_z_angle += z;
      
      // compensate for directional asymmetry in roll axis rate sensitivity
      //
      if (y > 0) GYRO_y_angle -= y / GYRO_comp;

      // smooth rate data using low pass filter
      //
      //   K    number of samples for output to reach 90% of input
      //  ---   --------------------------------------------------
      //   4     35 (= .35 sec @ 100 samples/sec)
      //   3     17
      //   2      7
      //   1      3
      //   0      1

      #define K 1 // filter strength (0=none, 1=weak, 4=strong)
      
      static SDWORD x_filter;
      x_filter = x_filter - (x_filter >> K) + x;
      GYRO_x_rate = x_filter >> K;

      static SDWORD y_filter;
      y_filter = y_filter - (y_filter >> K) + y;
      GYRO_y_rate = y_filter >> K;

      static SDWORD z_filter;
      z_filter = z_filter - (z_filter >> K) + z;
      GYRO_z_rate = z_filter >> K;
      
      GYRO_cnt += 1;
      }

   if (status & 0x80)
      { // gyro data changed before we had a chance to read it
      GYRO_misses += 1;
      }

   GYRO_ticks += 1;
   }

// --------------------------------------------------------------------
//                        Interface.
// --------------------------------------------------------------------

// Prepare gyro for use.
//
static void
GYRO_init()
   {
   // wait after powerup for gyro to finish loading its eeprom values (datasheet says 10ms, we'll use 20ms to be sure)
   delay_ms(20);
   
   // activate twi interface
   TWI_init();

   // 100Hz data rate, 12.5Hz bandwidth, power on, all axes enabled
   TWI_write(GYRO_ADDRESS, L3G_CTRL_REG1, 0x0F); // 0000.1111

   // enable block (atomic) updates to LSB,MSB data pairs
   TWI_write(GYRO_ADDRESS, L3G_CTRL_REG4, 0x80); // 1000.0000

#if 0
   BYTE who = TWI_read(GYRO_ADDRESS, L3G_WHO_AM_I);
   printf("who=%02x\n", who);
   for (;;);
#endif

   // begin collecting data
   TICKER_init(GYRO_interrupt);
   }

// Get elapsed time.
//
GYRO_TICKS
GYRO_getTicks()
   {
#if 0
   // read atomically by disabling interrupts
      DI();
      GYRO_TICKS t = GYRO_ticks;
      EI();
      return t;
#else
   // read atomically without disabling interrupts (to avoid starving integrator)
   for (;;)
      {
      GYRO_TICKS t = GYRO_ticks;
      // check if LSB rolled over into MSB(s) while we were reading the multi-byte value
      if ((t & 0xffffff00) == (GYRO_ticks & 0xffffff00))
         return t;
      }
#endif
   }

// Set gyro outputs to zero.
//
void
GYRO_zero()
   {
   DI();
   GYRO_ticks   = 0;
   GYRO_cnt     = 0;
   GYRO_x_angle = 0;
   GYRO_y_angle = 0;
   GYRO_z_angle = 0;
   EI();
   }

// Fast blink led for N seconds during gyro calibration.
//
static void
GYRO_blink(BYTE n)
   {
   n *= 8;
   while (n--)
      {
      LED_toggle();
      delay_ms(125);
      }
   }

// Get current die temperature.
// The indicated value is 0 at about 95 degrees F and decreases by 1 digit per degree C.
//
SBYTE
GYRO_temperature()
   {
   DI();
   BYTE t = TWI_read(GYRO_ADDRESS, L3G_OUT_TEMP);
   EI();
   return t;
   }

// Update zero rate biases to match current temperature.
//
void
GYRO_tc()
   {
   printf("tc\n");

   SBYTE t  = GYRO_temperature();
   SBYTE dT = t - GYRO_t_ref;
   
   GYRO_RATE x = GYRO_x_bias_ref + dT * Config.gyro_x_tc / 10;
   GYRO_RATE y = GYRO_y_bias_ref + dT * Config.gyro_y_tc / 10;
   GYRO_RATE z = GYRO_z_bias_ref + dT * Config.gyro_z_tc / 10;

#if 0
   printf("t: %d -> %d\n", GYRO_t, t);
   printf("x: %ld + %d * %d / 10 -> %ld\n", GYRO_x_bias_ref, Config.gyro_x_tc, dT, x);
   printf("y: %ld + %d * %d / 10 -> %ld\n", GYRO_y_bias_ref, Config.gyro_y_tc, dT, y);
   printf("z: %ld + %d * %d / 10 -> %ld\n", GYRO_z_bias_ref, Config.gyro_z_tc, dT, z);
#endif

#if 1
   printf("t: %d -> %d\n", GYRO_t_ref, t);
   printf("x: %ld -> %ld\n", GYRO_x_bias, x);
   printf("y: %ld -> %ld\n", GYRO_y_bias, y);
   printf("z: %ld -> %ld\n", GYRO_z_bias, z);
#endif
   
   DI();
   GYRO_x_bias = x;
   GYRO_y_bias = y;
   GYRO_z_bias = z;
   EI();
   }

// Read gyros for a few seconds and compute biases needed to zero the output rates.
// Assumption: device is motionless
//
void
GYRO_calibrate()
   {
   // integrate
   //
   GYRO_x_bias = 0;
   GYRO_y_bias = 0;
   GYRO_z_bias = 0;
   GYRO_zero();
   GYRO_blink(8);

   // calculate new biases
   //
   DI();
   SDWORD n = GYRO_cnt;
   GYRO_x_bias = GYRO_x_angle / n;
   GYRO_y_bias = GYRO_y_angle / n;
   GYRO_z_bias = GYRO_z_angle / n;
   EI();

   // remember them for use in temperature correction
   //
   GYRO_x_bias_ref = GYRO_x_bias;
   GYRO_y_bias_ref = GYRO_y_bias;
   GYRO_z_bias_ref = GYRO_z_bias;
   GYRO_t_ref      = GYRO_temperature();

   // output data in a format suitable for gnuplot temperature vs. zero rate bias graph
   //
   printf("%d %ld %ld %ld %ld\n", GYRO_t_ref, GYRO_x_bias_ref, GYRO_y_bias_ref, GYRO_z_bias_ref, n);
   }

// Converting gyro digits to rate and integrated angle.
//
// Let:
// R = sampling rate (samples per second)
// S = sensitivity (digits per degree-per-second)
//
// From the datasheet, the sampling rate is 100Hz and the gyro sensitivity is .00875 dps/digit:
// R = 100Hz
// S = 1 / (.00875 dps/digit) = 114.29 digits/dps
//
// The angular rate is therefore:
// angular-rate(tenths-of-a-degree-per-second) = 10 * angular-rate(digits) / sensitivity(digits/dps)
//                                             = 10 * angular-rate(digits) / 114.29
//                                             = 100 * angular-rate(digits) / 1143
// And the integrated angle is:
// integrated-digits = number-of-samples                         * angular-rate(dps) * sensitivity(digits/dps)
//                   = integration-time(sec) * R(/sec)           * angular-rate(dps) * S(digits/dps)
//                   = integration-time(sec) * angular-rate(dps) * RS(digits/dps/sec)
//                   = integrated-angle(deg)                     * RS(digits/deg)
// Thus:
// integrated-angle(deg) = integrated-digits / (R * S)
// integrated-angle(tenths-of-a-degree) = 10 * integrated-digits / (R * S)
//                                      = 10 * integrated-digits / (100 * 114.29)
//                                      = integrated-digits / 1143
//

// Convert gyro rate to tenths of a degree per second.
//
static DEGREES_PER_SECOND
GYRO_rate(GYRO_RATE r)
   { 
   return (DEGREES_PER_SECOND)( (r * 100) / Config.gyro_sensitivity );
   }

// Convert gyro angle to tenths of a degree.
//
static DEGREES
GYRO_angle(GYRO_ANGLE a)
   {
   return (DEGREES)(a / Config.gyro_sensitivity);
   }

#define GYRO_ROLL_BIAS   GYRO_y_bias
#define GYRO_YAW_BIAS    GYRO_x_bias
#define GYRO_PITCH_BIAS  GYRO_z_bias

#define GYRO_ROLL_RATE   GYRO_y_rate
#define GYRO_YAW_RATE    GYRO_x_rate
#define GYRO_PITCH_RATE  GYRO_z_rate

#define GYRO_ROLL_ANGLE  GYRO_y_angle
#define GYRO_YAW_ANGLE   GYRO_x_angle
#define GYRO_PITCH_ANGLE GYRO_z_angle

// Adjust gyro roll angle.
// Taken: multiplier (percent)
//
void
GYRO_adjustRollAngle(SWORD pct)
   {
   GYRO_ANGLE a;
   { DI(); a = GYRO_ROLL_ANGLE; EI(); }
   a = (a * pct) / 100;
   { DI(); GYRO_ROLL_ANGLE = a; EI(); }
   }

// Get current roll angle, in tenths of a degree.
//
DEGREES
GYRO_getRollAngle()
   {
   DI();
   GYRO_ANGLE angle = GYRO_ROLL_ANGLE;
   EI();
   return GYRO_angle(angle);
   }

// Get current roll rate, in tenths of a degree per second.
//
DEGREES_PER_SECOND
GYRO_getRollRate()
   {
   DI();
   GYRO_RATE rate = GYRO_ROLL_RATE;
   EI();
   return GYRO_rate(rate);
   }

// Get current yaw angle, in tenths of a degree.
//
DEGREES
GYRO_getYawAngle()
   {
   DI();
   GYRO_ANGLE angle = GYRO_YAW_ANGLE;
   EI();
   return GYRO_angle(angle);
   }

// Get current yaw rate, in tenths of a degree per second.
//
DEGREES_PER_SECOND
GYRO_getYawRate()
   {
   DI();
   GYRO_RATE rate = GYRO_YAW_RATE;
   EI();
   return GYRO_rate(rate);
   }

// Get current pitch angle, in tenths of a degree.
//
DEGREES
GYRO_getPitchAngle()
   {
   DI();
   GYRO_ANGLE angle = GYRO_PITCH_ANGLE;
   EI();
   return GYRO_angle(angle);
   }

// Get current pitch rate, in tenths of a degree per second.
//
DEGREES_PER_SECOND
GYRO_getPitchRate()
   {
   DI();
   GYRO_RATE rate = GYRO_PITCH_RATE;
   EI();
   return GYRO_rate(rate);
   }
