// Call a function periodically via hardware timer interrupt.
//
// Units:      TIMER0 channel A
// Interrupts: TIMER0_COMPA_vect
//
// Usage:      #define TICKS_PER_SEC 1000 // interrupt rate
//        /or/ #define TICKS_PER_SEC  200 // interrupt rate
//             #include "../include/ticker.h"

// Timer interrupt callback function.
//
typedef void (*TICKER_FUNC)();
static TICKER_FUNC TICKER_func;

// Interrupt service routine for TIMER0.
//
ISR(TIMER0_COMPA_vect)
   {
   (*TICKER_func)();
   }

// Set TIMER0 to call specified function at a rate of TICKS_PER_SEC.
// Taken:    function to be called
// Returned: nothing
//
// We use CTC ("clear on terminal count") wave generation mode.
// The counter continually runs from 0 to TOP and then resets to 0.
// An interrupt is generated each time TOP is reached.
//
// The TOP count needed for a given interrupt rate is:
//    TOP = (FREQ / PRESCALER / RATE) - 1             
//
// Conversely the interrupt rate is: 
//    RATE = FREQ / PRESCALER / (TOP + 1)
//
// For FREQ =  8Mhz, PRESCALER = 64, RATE = 1000Hz:
//    TOP = (8000000 / 64 / 1000) - 1 = 124
//
// For FREQ =  8Mhz, PRESCALER = 256, RATE = 200Hz:
//    TOP = (8000000 / 256 / 200) - 1 = 155.25 => 155  [ actual rate = 8000000 / 256 / (155 + 1) = 200.3 ]
//
static void
TICKER_init(TICKER_FUNC f)
   {
   TICKER_func = f;

   // set timer for "clear on terminal count" operation (mode 2)
   //
   TCCR0A = 0
          | (0 << WGM00) // CTC wave generation (mode 2)
          | (1 << WGM01) // CTC wave generation (mode 2)
          ;

   TCCR0B = 0
          | (0 << WGM02) // CTC wave generation (mode 2)
          ;

   // set terminal count value (TOP) at which to generate a
   // "compare match A" interrupt and clear the counter back to zero
   // note that TIMER0 is an 8 bit timer, so TOP must be <= 255
   //
#if CLOCK_MHZ != 8
#error clock!
#endif

#if   TICKS_PER_SEC == 1000
   OCR0A = 124;
#elif TICKS_PER_SEC == 200
   OCR0A = 155;
#else
#error ticks!
#endif
   
   // enable "TIMER0 compare match A" interrupts
   //
   TIMSK0 |= (1 << OCIE0A);

   // start the timer by enabling the prescaler
   //
   TCCR0B = 0
#if   TICKS_PER_SEC == 1000
          | (1 << CS00)  // timer clock = system clock / 64
          | (1 << CS01)  // "
          | (0 << CS02)  // "
#elif TICKS_PER_SEC == 200
          | (0 << CS00)  // timer clock = system clock /  256
          | (0 << CS01)  // "
          | (1 << CS02)  // "
#else
#error ticks!
#endif
          ;
   }

// Reset interrupt upcounter.
//
void
TICKER_reset()
   {
   TCNT0 = 0;
   }
