// A motorcycle gyrocam controller.
// 25 Sep 2012 Derek Lieber
//
// Hardware: Atmel ATMEGA328P mcu
//           STmicroelectronics L3G-4200D or L3G-D20 three axis gyro chip (Pololu)
//           Hitec HS-425BB servo (Servo City)
//           Contour Roam camera with lens barrel dust seal & clickers removed
//

#define CLOCK_INTERNAL 1           // internal oscillator
#define CLOCK_MHZ      8           // 8 Mhz system clock

#include <avr/io.h>                // avr architecture - see /usr/lib/a./include/avr/iomx8.h
#include <avr/interrupt.h>         // avr interrupt helpers - ISR, sei, cli
#include "./include/types.h"      // BOOL, BYTE, WORD, DWORD, FLOAT
#include "./include/system.h"     // standard startup
#include "./include/atomic.h"     // EI DI
#include "./include/led.h"        // status led
#include "./include/usart.h"      // serial i/o via usart
#include "./include/stdout.h"     // stdout via usart
#include "./include/eeprom.h"     // persistent memory
#include "./include/reset.h"      // procesor reset
#include "./include/bootloader.h" // LOADER_REQUEST_REBOOT
#define  TICKS_PER_SEC 200         // timer tick interrupt rate
#include "./include/ticker.h"     // timer tick interrupt
#include "./include/twi.h"        // two-wire interface
#include "./include/delay.h"      // delay_ms

#include <stdlib.h>                // abs()

#include "version.h"               // date of issue
#include "types.h"                 // application-specific data types
#include "config.h"                // configuration constants
#include "stack.h"                 // stack checker
#include "battery.h"               // battery monitor
#include "gyro.h"                  // gyro controller
#include "servo.h"                 // servo controller
#include "camera.h"                // camera controller

//----------------------------------------------------------------------
// Atmega328 pin and port allocations.
//----------------------------------------------------------------------
//
//                        AREF = pin 21
//                        AVCC = pin 20 <- 5v
//                         VCC = pin  7 <- 5v
//                        AGND = pin 22 <- 0v
//                         GND = pin  8 <- 0v
//                      #RESET = pin  1 <- 10k pullup + grounding button
//
// [ICP1][PCINT0][CLKO] PORTB0 = pin 14 
// [OC1A][PCINT1]       PORTB1 = pin 15 -> servo
// [OC1B][PCINT2] [#SS ]PORTB2 = pin 16
// [OC2A][PCINT3] [MOSI]PORTB3 = pin 17
//       [PCINT4] [MISO]PORTB4 = pin 18
//       [PCINT5] [SCK ]PORTB5 = pin 19
//       [PCINT6][XTAL1]PORTB6 = pin  9
//       [PCINT7][XTAL2]PORTB7 = pin 10
//
//       [PCINT8] [ADC0]PORTC0 = pin 23 -> status led
//       [PCINT9] [ADC1]PORTC1 = pin 24 <- battery voltage divider
//       [PCINT10][ADC2]PORTC2 = pin 25 
//       [PCINT11][ADC3]PORTC3 = pin 26 
// [SDA] [PCINT12][ADC4]PORTC4 = pin 27 <-> gyro SDA
// [SCL] [PCINT13][ADC5]PORTC5 = pin 28 ->  gyro SCL
//                      PORTC6 = n/a
//                      PORTC7 = n/a
//
//       [PCINT16] [RXD]PORTD0 = pin  2 <- serial connection + 10k pulluup
//       [PCINT17] [TXD]PORTD1 = pin  3 -> serial connection
//       [PCINT18][INT0]PORTD2 = pin  4 
// [OC2B][PCINT19][INT1]PORTD3 = pin  5 
// [XCK] [PCINT20]  [T0]PORTD4 = pin  6 
// [OC0B][PCINT21]  [T1]PORTD5 = pin 11 
// [OC0A][PCINT22][AIN0]PORTD6 = pin 12 
//       [PCINT23][AIN1]PORTD7 = pin 13 

//---------------------------------------------------------------------
//               Begin Hardware Testing And Calibration
//---------------------------------------------------------------------

void
adjust_led()
   {
   for (;;)
      {
      switch (USART_get())
         {
         case '1': LED_on();     break;
         case '0': LED_off();    break;
         default:  LED_toggle(); break;
         case 'q': printf("\n"); LED_off(); return;
         }
      }
   }
         
void
adjust_battery()
   {
   for (;;)
      {
      WORD val = ADC_read_battery();
      WORD mv  =  ADC_battery_counts_to_millivolts(val);
      WORD pct = (mv - 7400) * 100L / (8400 - 7400); // 2s lipo is 7.4V to 8.4V (curve is not really linear, but close)
      printf("0x%03x %4u %4umV %3u%% k=%4lu\r", val, val, mv, pct, Config.battery_divisor);
      switch (USART_get())
         {
         case '+': case '=': Config.battery_divisor -= 1; break; // raise indicated voltage
         case '-':           Config.battery_divisor += 1; break; // lower indicated voltage

         case 'q': printf("\n"); return;
         default:  printf("\n?\n");
         }
      }
   }
         
void
adjust_servo()
   {
   for (;;)
      {
      printf("%6d b=%6d\r", SERVO_getAngle(), Config.servo_bias);
      switch (USART_get())
         {
         // fine step
         case 'j': SERVO_setAngle(SERVO_getAngle() -  1 * 10); break;
         case 'k': SERVO_setAngle(SERVO_getAngle() +  1 * 10); break;
         
         // half throw
         case 'h': SERVO_setAngle(- 45 * 10); break;
         case 'l': SERVO_setAngle(+ 45 * 10); break;

         // full throw
         case 'H': SERVO_setAngle(- 90 * 10); break;
         case 'L': SERVO_setAngle(+ 90 * 10); break;
         
         // no throw
         case 'z': SERVO_setAngle(0); break;

         // trim to horizon
         case '+': 
         case '=': Config.servo_bias += 5;       SERVO_setAngle(0); break;
         case '-': Config.servo_bias -= 5;       SERVO_setAngle(0); break;
         
         case 'q': printf("\n"); return;
         default:  printf("\n?\n");
         }
      }
   }

void
adjust_camera()
   {
   BOOL track = 0;
   BOOL tc = 0;
   for (;;)
      {
      while (!USART_ready())
         {
         if (track)
            CAMERA_setAngle(GYRO_getRollAngle());
         
         printf("gear=%3ld/%3ld T=%3dC tc=%d roll: bias=%4ld comp=%4ld angle=%4d rate=%6d yaw: rate=%6d\r", CAMERA_GEAR_POS, CAMERA_GEAR_NEG, GYRO_temperature(), tc, GYRO_ROLL_BIAS, GYRO_comp, GYRO_getRollAngle(), GYRO_getRollRate(), GYRO_getYawRate() );
         
         static GYRO_TICKS start;
         GYRO_TICKS now = GYRO_getTicks();
         if (now - start > TICKS_PER_SEC * 10)
            {
            if (tc) GYRO_tc();
            start = now;
            }
         }
         
      switch (USART_get())
         {
         // camera/servo drive ratio
         case 'P': CAMERA_GEAR_POS += 1; break;
         case 'p': CAMERA_GEAR_POS -= 1; break;
         
         case 'N': CAMERA_GEAR_NEG += 1; break;
         case 'n': CAMERA_GEAR_NEG -= 1; break;
         
         // roll axis bias adjustment
         case '+': 
         case '=': GYRO_ROLL_BIAS -= 1; break; // increase zero rate
         case '-': GYRO_ROLL_BIAS += 1; break; // decrease zero rate
         
         // roll axis asymmetry compensation
         case 'R': GYRO_comp += 10; break;
         case 'r': GYRO_comp -= 10; break;
         
         case '.': track = !track; break;
         case 't': tc = !tc; break;
         case 'z': printf("\n"); GYRO_zero(); break;
         case 'c': printf("\n"); GYRO_calibrate(); GYRO_zero(); break;
         
         // generate zero-bias vs. temperature data for gnuplot
         case 'd': while (!USART_ready())
                      {
                      GYRO_calibrate();
                      delay_ms(10*1000);
                      }
                   break;
      
         // generate stability data for gnuplot
         case 'D': while (!USART_ready())
                      {
                      DEGREES roll  = GYRO_getRollAngle();
                      DEGREES yaw   = GYRO_getYawAngle();
                      DEGREES pitch = GYRO_getPitchAngle();
                      SBYTE   t     = GYRO_temperature();
                      printf("%d %d %d %d\n", t, roll, yaw, pitch);
                      delay_ms(10*1000);
                      if (tc) GYRO_tc();
                      }
                   break;
      
         case 'q': printf("\n"); return;
         default:  printf("\n?\n");
         }
      }
   }
         
void
adjust_gyro()
   {
   for (;;)
      {
      while (!USART_ready())
         {
#if 1
         printf("t=%6lu c=%6lu m=%u "
                "rx=%5d ry=%5d rz=%5d "
                "x=%5d y=%5d z=%5d "
                "k=%ld "
                "\r",
                GYRO_ticks,
                GYRO_cnt,
                GYRO_misses,
                GYRO_rate(GYRO_x_rate), GYRO_rate(GYRO_y_rate), GYRO_rate(GYRO_z_rate),
                GYRO_angle(GYRO_x_angle), GYRO_angle(GYRO_y_angle), GYRO_angle(GYRO_z_angle),
                Config.gyro_sensitivity
                );
#else
         printf("t=%6lu c=%6lu m=%u "
                "rr=%5d ry=%5d rp=%5d "
                "r=%5d y=%5d p=%5d "
                "k=%ld "
                "\r",
                GYRO_ticks,
                GYRO_cnt,
                GYRO_misses,
                GYRO_getRollRate(), GYRO_getYawRate(), GYRO_getPitchRate(),
                GYRO_getRollAngle(), GYRO_getYawAngle(), GYRO_getPitchAngle(),
                Config.gyro_sensitivity
                );
#endif
         }
      
      switch (USART_get())
         {
         case '+': 
         case '=': Config.gyro_sensitivity -= 1; break; // increase angle
         case '-': Config.gyro_sensitivity += 1; break; // decrease angle
         case 'z': GYRO_zero(); break;
         case 'c': printf("\n"); GYRO_calibrate(); GYRO_zero(); break;
         case 'q': printf("\n"); return;
         default:  printf("\n?\n");
         }
      }
   }

//---------------------------------------------------------------------
//                 End Hardware Testing And Calibration
//---------------------------------------------------------------------

// Handle a TWI failure.
// Taken:    operation type (start, send, recv, stop)
//           reported TWI status
//           expected TWI status
// Returned: does not return
//
void 
TWI_fail(const char *type, BYTE status, BYTE expected)
   {
   // stop taking measurements
   DI();
   
   // move camera to failsafe position
   SERVO_slew(0);
   
   // report problem
   printf("twi %s failed: status=0x%02x expected=0x%02x\n", type, status, expected);
   for (;;)
      {
      LED_on();  delay_ms(125);
      LED_off(); delay_ms(125);
      LED_on();  delay_ms(125);
      LED_off(); delay_ms(125);
      delay_ms(500);
      }
   
   // not reached
   (void)sreg;
   }

// Mainline.
//
void
run()
   {
   // indicate running
   //
   LED_on();
   
   // prepare gyro
   //
   printf("cal\n");
   GYRO_blink(3); // let power stabilize after servo turn-on glitch
   GYRO_calibrate();
   GYRO_zero();
   
   // drive camera, perform temperature compensation, and monitor battery
   //
   for (;;)
      {
      while (!USART_ready())
         {
      #if 0
         // measure update rate [ note: must link with floating point print library ]
         // I see: ~800 updates/sec
         //
         {
         static GYRO_TICKS start;
         static WORD  n;
         if (++n == 500)
            {
            GYRO_TICKS now = GYRO_getTicks();
            GYRO_TICKS elapsed = now - start;
            printf("%d updates, %lu ticks, %u ticks/second, %.2f updates/second\n", n, elapsed, TICKS_PER_SEC, n / ( (FLOAT)elapsed / TICKS_PER_SEC ));
            n = 0;
            start = now;
            }
         }
      #endif
         
      #if 1
         // update zero rate biases to match temperature
         //
         {
         static GYRO_TICKS start;
         GYRO_TICKS now = GYRO_getTicks();
         if (now - start > TICKS_PER_SEC * 10)
            {
            GYRO_tc();
            start = now;
            }
         }
      #endif
         
         // match camera angle to gyro
         //
         CAMERA_setAngle(GYRO_getRollAngle());

      #if 1
         // align gyro and camera to inferred horizon
         //
         #define GYRO_LOW  5 // .5 deg/sec
         if (abs(GYRO_getRollRate()) < GYRO_LOW && abs(GYRO_getYawRate()) < GYRO_LOW)
            { 
            GYRO_adjustRollAngle(0);
            SERVO_slew(0);
            }
      #endif

         // monitor battery voltage
         //
         if (BAT_read() < 7650) // 7.65V = 3.82V/cell = 25% charge
            { // blink low battery warning
            static GYRO_TICKS start;
            GYRO_TICKS now = GYRO_getTicks();
            if (now - start > TICKS_PER_SEC / 2)
               {
               LED_toggle();
               start = now;
               }
            }
         else
            LED_on();
         }
   
      switch (USART_get())
         {
         case LOADER_REQUEST_REBOOT: reset_processor(); break;
         case 'q': printf("\n"); return;
         default:  printf("?\n");
         }
      }
   }

int
main()
   {
// BOOL is_software_reset = (MCUSR == 0);
   MCUSR = 0;
   
   STACK_init();
   SYSTEM_init();
   USART_init();
   STDOUT_init();
   LED_init();
   
   printf("%s\n", Version);

   CONFIG_load();

#if 1 // normal startup
   BAT_init();
   SERVO_init();
   GYRO_init();
   run();
#else // hardware testing
   (void)is_software_reset;
   (void)GYRO_init;
   BAT_init();
   SERVO_init();
   adjust_servo();
#endif

   for (;;)
      {
      printf("[%u] I=init D=dump S=save l=led b=battery v=servo c=camera g=gyro r=run f=fail %c=reboot>", STACK_free(), LOADER_REQUEST_REBOOT);

      char ch = USART_get();
      printf("\n");
      
      switch (ch)
         {
         case 'I': CONFIG_init();    break;
         case 'D': CONFIG_dump();    break;
         case 'S': CONFIG_save();    break;
         case 'l': adjust_led();     break;
         case 'b': adjust_battery(); break;
         case 'v': adjust_servo();   break;
         case 'c': adjust_camera();  break;
         case 'g': adjust_gyro();    break;
         case 'r': run();            break;
         case 'f': TWI_fail("no op", 0, 0); break;
         
         case LOADER_REQUEST_REBOOT: reset_processor(); break;
                   
         default:  printf("?\n"); break;
         }
      }

   return 0;
   }
