#if HAVE_ACCELEROMETERS

// STmicroelectronics LSM303DLHC 3-axis accelerometer.
//
// Accelerometer characteristics:
//
//    range        +/- 2 gee
//    resolution   12 bits
//    sensitivity  .001 gee per digit
//    update rate  1/10/25/50/100/200/400 Hz

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

// Types.
//
typedef SWORD ACC_DIGITS; // an unscaled accelerometer reading [range=-2k..+2k, nominal sensitivity=.001 gee per digit]

// Registers.
//
#define ACC_CTRL_REG1_A  0x20
#define ACC_CTRL_REG2_A  0x21
#define ACC_CTRL_REG3_A  0x22
#define ACC_CTRL_REG4_A  0x23
#define ACC_CTRL_REG5_A  0x24
#define ACC_CTRL_REG6_A  0x25
#define ACC_REFERENCE_A  0x26
#define ACC_STATUS_REG_A 0x27

#define ACC_OUT_X_L_A    0x28
#define ACC_OUT_X_H_A    0x29
#define ACC_OUT_Y_L_A    0x2A
#define ACC_OUT_Y_H_A    0x2B
#define ACC_OUT_Z_L_A    0x2C
#define ACC_OUT_Z_H_A    0x2D

// Access.
//
#define ACC_ADDR 0x19

// Read accelerometer sensors, mapping sensor axes to body axes such that:
//    x points ahead (body roll axis)
//    y points right (body pitch axis)
//    z points down  (body yaw axis)
//
static void
ACC_read_xyz(ACC_DIGITS *x, ACC_DIGITS *y, ACC_DIGITS *z)
   {
   BYTE b[6];
   TWI_read_multi(ACC_ADDR, ACC_OUT_X_L_A | 0x80, sizeof(b), b); // 0x80 is autoincrement bit

   *z =  (((b[1] << 8) | b[0]) >> 4); // X sensor
   *x =  (((b[3] << 8) | b[2]) >> 4); // Y sensor
   *y = -(((b[5] << 8) | b[4]) >> 4); // Z sensor
   }

#define ACC_X_BIAS   Config.acc_x_bias
#define ACC_Y_BIAS   Config.acc_y_bias
#define ACC_Z_BIAS   Config.acc_z_bias

#define ACC_X_GAIN   Config.acc_x_gain
#define ACC_Y_GAIN   Config.acc_y_gain
#define ACC_Z_GAIN   Config.acc_z_gain

// Smoothing constant.
//
//             measured
//    K    settling time @ 50Hz
//    -    --------------------
//    6        10sec
//    5         5sec
//    4         3sec
//    3         1sec
//
#define ACC_FILTER_K Config.acc_filter_k

//------------------------------
// Interrupt communication area.
//
static volatile WORD       ACC_hits;   // number of times data changed
static volatile WORD       ACC_misses; // number of times data changed before we could read it (should only happen once, at startup)

static volatile ACC_DIGITS ACC_x;      // bias corrected and smoothed accelerometer reading, updated at ACC_HZ rate
static volatile ACC_DIGITS ACC_y;      // "
static volatile ACC_DIGITS ACC_z;      // "
//
//------------------------------

// Update accelerometers.
// Called by interrupt.
//
static inline void
ACC_update()
   {
   BYTE status = TWI_read(ACC_ADDR, ACC_STATUS_REG_A);
   
   if (status & 8)
      { // new data is available (note: timer tick interrupt fires at a rate faster than accelerometer data changes)
      
      // raw sensor readings
      //
      ACC_DIGITS x, y, z;
      ACC_read_xyz(&x, &y, &z);
      
      // remove zero rate biases
      //
      x -= ACC_X_BIAS;
      y -= ACC_Y_BIAS;
      z -= ACC_Z_BIAS;

      // smooth with a low pass filter
      // K = filter strength (0=none, 1=weak, 4+=strong)
      //
      const BYTE K = ACC_FILTER_K;
      
      static SDWORD x_filter;
      x_filter = x_filter - (x_filter >> K) + x;
      ACC_x = x_filter >> K;

      static SDWORD y_filter;
      y_filter = y_filter - (y_filter >> K) + y;
      ACC_y = y_filter >> K;

      static SDWORD z_filter;
      z_filter = z_filter - (z_filter >> K) + z;
      ACC_z = z_filter >> K;

      ACC_hits += 1;
      }

   if (status & 0x80)
      { // data changed before we had a chance to read it
      ACC_misses += 1;
      }
   }
   
// --------------------------------------------------------------------
//                          Interface.
// --------------------------------------------------------------------

// Prepare accelerometers for use.
//
static void
ACC_init()
   {
   #if ACC_HZ != 50
   #error ACC_HZ
   #endif
   
   // 50Hz data rate, power on, enable all axes
   TWI_write(ACC_ADDR, ACC_CTRL_REG1_A, 0x47); // 0100.0111
   
   // enable block (atomic) updates to LSB,MSB data pairs, 2 gee scale, hi resolution (more data smoothing, compared to lo resolution)
   TWI_write(ACC_ADDR, ACC_CTRL_REG4_A, 0x88); // 1000.1000

   // read sensors once to restart adc after changing settings
   ACC_DIGITS x, y, z;
   while ((TWI_read(ACC_ADDR, ACC_STATUS_REG_A) & 8) == 0) ;
   ACC_read_xyz(&x, &y, &z);
   }

// Get (smoothed) accelerometer readings, in gees.
//
FLOAT
ACC_getX()
   {
   DI();
   ACC_DIGITS x = ACC_x;
   EI();
   return x * ACC_X_GAIN;
   }

FLOAT
ACC_getY()
   {
   DI();
   ACC_DIGITS y = ACC_y;
   EI();
   return y * ACC_Y_GAIN;
   }

FLOAT
ACC_getZ()
   {
   DI();
   ACC_DIGITS z = ACC_z;
   EI();
   return z * ACC_Z_GAIN;
   }

// Get magnitude of accelerometer vector, in gees.
// This is the actual g-force felt by the camera, bike, and rider.
//
//      value     example
//      ------    -------
//      g =  0   free fall
//      g <  1   descending or falling
//      g == 1   constant velocity
//      g >  1   climbing, turning, throttling, braking
//
FLOAT
ACC_getG()
   {
   FLOAT x = ACC_getX();
   FLOAT y = ACC_getY();
   FLOAT z = ACC_getZ();
   return sqrt(x * x + y * y + z * z);
   }

// Get direction of accelerometer vector with respect to ground reference frame, in radians.
// Note that these angles are only meaningful when the sensor is moving at constant velocity, ie. when ACC_getG() == 1.
// Ref: http://www.freescale.com/files/sensors/doc/app_note/AN3461.pdf [Eqn 25, 26]
//
FLOAT
ACC_getRollAngle()
   {
   FLOAT y = ACC_getY();
   FLOAT z = ACC_getZ();

   return atan2(y, z);
   }

FLOAT
ACC_getPitchAngle()
   {
   FLOAT x = ACC_getX();
   FLOAT y = ACC_getY();
   FLOAT z = ACC_getZ();

   return atan2(-x, sqrt(y * y + z * z));
   }

#else

WORD  ACC_hits;
WORD  ACC_misses;
BYTE  ACC_FILTER_K;
void  ACC_init()   {}
void  ACC_update() {}
FLOAT ACC_getRollAngle()  { return 0; }
FLOAT ACC_getPitchAngle() { return 0; }
FLOAT ACC_getX()          { return 0; }

#endif
