// Battery monitor.
//
// Units:      ADC
// Interrupts: none
// Pins:       ADC1 (PORTC1)
// Clock:      8Mhz
//

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

#define ADC_MAX_COUNT 0x3ff  // range is 0..1023

// Take a reading on currently selected adc.
// Taken:    nothing
// Returned: adc counts (0..ADC_MAX_COUNT)
//
static WORD
ADC_read()
   {
   // start adc converter
   //
   ADCSRA |= (1 << ADSC);

   // wait for conversion to complete
   //
   while (ADCSRA & (1 << ADSC));

   // read out the result
   //
   BYTE lsb = ADCL;
   BYTE msb = ADCH;
   WORD val = (msb << 8) | lsb;
   return val;
   }

// Take a reading on ADC1 (external battery monitor).
// Taken:    nothing
// Returned: adc counts (0..ADC_MAX_COUNT)
//
static WORD
ADC_read_battery()
   {
   // select ADC1
   //
   ADMUX  = 0             
          | (1 << REFS0)  // voltage reference = internal 1.1v
          | (1 << REFS1)

          | (1 << MUX0)   // voltage source = ADC1
          | (0 << MUX1)
          | (0 << MUX2)
          | (0 << MUX3)
          ;
   
   // use it
   //
   return ADC_read();
   }

// Convert adc reading to volts.
//
static FLOAT
ADC_battery_counts_to_volts(WORD adc_counts)
   {
   return adc_counts * Config.battery_multiplier;
   }

// --------------------------------------------------------------------
//                          Interface.
// --------------------------------------------------------------------

// Prepare adc for use.
//
static void
BAT_init()
   {
   // adc needs a 50-200kHz clock
   // lower clock  = more accurate
   // higher clock = faster conversion
   // for system clock of 8Mhz:
   //    prescaler  adc clock
   //       128      62.5kHz
   //       64       125 kHz
   //       32       250 kHz

   #if CLOCK_MHZ != 8
   #error CLOCK_MHZ
   #endif
   
   ADCSRA = 0
          | (1 << ADPS0)  // adc prescaler = divide by 128
          | (1 << ADPS1)
          | (1 << ADPS2)
          
          | (1 << ADEN)   // enable adc
          ;

   // run adc once to put it in a clean state
   ADC_read(); 
   }

// Get battery reading, in volts.
//
FLOAT
BAT_read()
   {
   return ADC_battery_counts_to_volts(ADC_read_battery());
   }

//                   LIPO Resting Voltages
// --------------------------------------------------------------
// 1 cell pack     2 cell pack     3 cell pack     4 cell pack
// 4.20v = 100%    8.40v = 100%    12.60v = 100%   16.80v = 100%
// 4.03v = 76%     8.06v = 76%     12.09v = 76%    16.12v = 76%
// 3.86v = 52%     7.72v = 52%     11.58v = 52%    15.44v = 52%
// 3.83v = 42%     7.66v = 42%     11.49v = 42%    15.32v = 42%
// 3.79v = 30%     7.58v = 30%     11.37v = 30%    15.16v = 30%
// 3.70v = 11%     7.40v = 11%     11.10v = 11%    14.80v = 11%
// 3.6?v = 00%     7.2?v = 00%     10.8?v = 00%    14.4?v = 00%

#define BATTERY_LOW 7.60 // battery low voltage cutoff threshold (30%)

// Has battery voltage remained lower than cutoff threshold for longer than 1 second?
//
BOOL
BAT_low()
   {
   if (BAT_read() > BATTERY_LOW)
      return 0; // battery ok

   TICKS start = TIME_now();
   for (;;)
      {
      if (BAT_read() > BATTERY_LOW)
         return 0;  // battery bounced back
      if (TIME_elapsed(&start) >= 1)
         return 1;  // battery still low after 1sec
      }
   }
