// Calibration and tuning parameters.
//
// The "baseline" configuration values (CAMERA_ID=99) seem to work ok but they can be adjusted to improve accuracy, as follows.
//
// 1. Set accelerometer biases and gains for all three axes. See adjust_acc().
//
// 2. Set gyro gains such that angles derived from imu match those derived from accelerometer 
//    for each of six rotational directions. See adjust_gyro().
//
// 3. Set camera servo bias (C) to align camera's laser to horizon.
//    Set camera servo gains (L,R) such that laser tracks horizon through +/-45 degree rolls.
//    See adjust_camera().
//
// 4. Set battery multiplier such that readout matches external voltmeter. See adjust_battery().
//
// 5. Set gyro temperature compensation constants by curve fitting rate vs temperature data with gnuplot. See plot_tc().
//    Example:
//      ------ curve fitting -------   ------- constants -----
//      x-out = ( 89.1) + (-5.1) * t   Config.gyro_x_tc = -5.1;
//      y-out = ( 16.2) + ( 0.1) * t   Config.gyro_y_tc =   .1;
//      z-out = (-99.2) + ( 2.0) * t   Config.gyro_z_tc =  2.0;
//
// 6. Choose gyro, accelerometer, and imu smoothing and drift correction constants based on data visualization with gnuplot and actual road tests.
//    See adjust_imu().
//
typedef struct 
   {
   // calibration parameters
   //
   SWORD acc_x_bias, acc_y_bias, acc_z_bias; // accelerometer zero biases
   FLOAT acc_x_gain, acc_y_gain, acc_z_gain; // accelerometer gains to yield 1 gee on each axis

   FLOAT gyro_gain_xpos, gyro_gain_xneg;     // gyro gains in each of the six rotational directions...
   FLOAT gyro_gain_ypos, gyro_gain_yneg;     // ...in radians/sec per digit...
   FLOAT gyro_gain_zpos, gyro_gain_zneg;     // ...adjusted such that imu-generated angles match accelerometer-generated angles

   FLOAT cam_bias;                           // camera servo centering bias, in degrees
   FLOAT cam_l_gain, cam_r_gain;             // camera servo gain in left and right directions

   FLOAT battery_multiplier;                 // battery adc-digits to volts conversion factor

   FLOAT gyro_x_tc, gyro_y_tc, gyro_z_tc;    // gyro temperature coefficients, in digits per degree C
   
   // tuning parameters
   //
   BYTE  gyro_filter_k;                      // gyro data smoothing strength
   BYTE  acc_filter_k;                       // accelerometer data smoothing strength
   BYTE  servo_filter_k;                     // servo output smoothing strength
   FLOAT imu_rate_threshold;                 // drift correction snapshots are taken whenever turn rate is lower than this, in radians/sec
   FLOAT imu_rate_duration;                  // ...for at least this long, in seconds
   FLOAT imu_time_constant;                  // time constant characterizing speed with which drift corrections are applied, in seconds
   } CONFIG;

CONFIG Config;

// Initialize.
//
static void
CONFIG_init()
   {
   //
   // calibration parameters
   //
   
//---------------------------------------------------------------------------------------------------------------------------------------------------
#if CAMERA_ID == 99 // baseline values

   #define HAVE_ACCELEROMETERS 0
   #define HAVE_LCD_DISPLAY    0
   #define GYRO_TWI_ADDRESS    0x69       // Pololu L3G4200D
// #define GYRO_TWI_ADDRESS    0x6B       // Pololu L3GD20
// #define GYRO_TWI_ADDRESS    0x6B       // Pololu MiniIMU
 
   Config.acc_x_bias         = 0;
   Config.acc_y_bias         = 0;
   Config.acc_z_bias         = 0;

   Config.acc_x_gain         = 0.001;
   Config.acc_y_gain         = 0.001;
   Config.acc_z_gain         = 0.001;

   Config.gyro_gain_xpos     = 0.0001527; // 0.0001527rad/sec = .00875deg/sec = datasheet value
   Config.gyro_gain_xneg     = 0.0001527;

   Config.gyro_gain_ypos     = 0.0001527;
   Config.gyro_gain_yneg     = 0.0001527;

   Config.gyro_gain_zpos     = 0.0001527;
   Config.gyro_gain_zneg     = 0.0001527;

   Config.cam_bias           =  0.0;
   Config.cam_l_gain         = 15.0;      // 1.5:1 gear ratio
   Config.cam_r_gain         = 15.0;

   Config.battery_multiplier = 0.01000;

   Config.gyro_x_tc          = 0.0;
   Config.gyro_y_tc          = 0.0;
   Config.gyro_z_tc          = 0.0;

//---------------------------------------------------------------------------------------------------------------------------------------------------
#elif CAMERA_ID == 1 // my test mule

   #define HAVE_ACCELEROMETERS 1
   #define HAVE_LCD_DISPLAY    1
   #define GYRO_TWI_ADDRESS    0x6B // Pololu L3GD20
 
   Config.acc_x_bias         =   1;
   Config.acc_y_bias         =  12;
   Config.acc_z_bias         = -19;

   Config.acc_x_gain         = 0.0009611;
   Config.acc_y_gain         = 0.0009886;
   Config.acc_z_gain         = 0.0009945;

   Config.gyro_gain_xpos     = 0.0001520;
   Config.gyro_gain_xneg     = 0.0001520;

   Config.gyro_gain_ypos     = 0.0001565;
   Config.gyro_gain_yneg     = 0.0001505;

   Config.gyro_gain_zpos     = 0.0001495;
   Config.gyro_gain_zneg     = 0.0001490;

   Config.cam_bias           = -1.9;
   Config.cam_l_gain         = 10.5;
   Config.cam_r_gain         = 10.3;

   Config.battery_multiplier = 0.00999;

   Config.gyro_x_tc          = -3.7;
   Config.gyro_y_tc          =  8.4;
   Config.gyro_z_tc          = -4.0;

//---------------------------------------------------------------------------------------------------------------------------------------------------
#elif CAMERA_ID == 2 // Jamie's bike

   #define HAVE_ACCELEROMETERS 0
   #define HAVE_LCD_DISPLAY    0
   #define GYRO_TWI_ADDRESS    0x69 // Pololu L3G4200D

   Config.gyro_gain_xpos     = 0.0001547;
   Config.gyro_gain_xneg     = 0.0001547;

   Config.gyro_gain_ypos     = 0.0001532;
   Config.gyro_gain_yneg     = 0.0001567;

   Config.gyro_gain_zpos     = 0.0001507;
   Config.gyro_gain_zneg     = 0.0001497;

   Config.cam_bias           = -2.7;
   Config.cam_l_gain         = 15.9;
   Config.cam_r_gain         = 15.6;

   Config.battery_multiplier = 0.01021;

   Config.gyro_x_tc          = -0.2;
   Config.gyro_y_tc          =  1.4;
   Config.gyro_z_tc          =  5.2;

//---------------------------------------------------------------------------------------------------------------------------------------------------
#elif CAMERA_ID == 3 // crashed, sensor destroyed

   #define HAVE_ACCELEROMETERS 0
   #define HAVE_LCD_DISPLAY    0
   #define GYRO_TWI_ADDRESS    0x6B // Pololu L3GD20

   Config.gyro_gain_xpos     = 0.0001547;
   Config.gyro_gain_xneg     = 0.0001552;

   Config.gyro_gain_ypos     = 0.0001542;
   Config.gyro_gain_yneg     = 0.0001552;

   Config.gyro_gain_zpos     = 0.0001602;
   Config.gyro_gain_zneg     = 0.0001602;

   Config.cam_bias           = -2.5;
   Config.cam_l_gain         = 15.3;
   Config.cam_r_gain         = 15.0;

   Config.battery_multiplier = 0.00982;

   Config.gyro_x_tc          = -2.7;
   Config.gyro_y_tc          =  9.5;
   Config.gyro_z_tc          = -2.0;

//---------------------------------------------------------------------------------------------------------------------------------------------------
#elif CAMERA_ID == 4  // Palmer's bike

   #define HAVE_ACCELEROMETERS 0
   #define HAVE_LCD_DISPLAY    0
   #define GYRO_TWI_ADDRESS    0x69 // Pololu L3G4200D

   Config.gyro_gain_xpos     = 0.0001492;
   Config.gyro_gain_xneg     = 0.0001482;

   Config.gyro_gain_ypos     = 0.0001527;
   Config.gyro_gain_yneg     = 0.0001542;

   Config.gyro_gain_zpos     = 0.0001482;
   Config.gyro_gain_zneg     = 0.0001482;

   Config.cam_bias           =  1.2;
   Config.cam_l_gain         = 15.6;
   Config.cam_r_gain         = 15.8;

   Config.battery_multiplier = 0.01011;

   Config.gyro_x_tc          =  1.2;
   Config.gyro_y_tc          =  0.6;
   Config.gyro_z_tc          =  4.3;

//---------------------------------------------------------------------------------------------------------------------------------------------------
#elif CAMERA_ID == 5 // replacement for crashed cam 3

   #define HAVE_ACCELEROMETERS 0
   #define HAVE_LCD_DISPLAY    0
   #define GYRO_TWI_ADDRESS    0x6B // Pololu L3GD20
 
   Config.gyro_gain_xpos     = 0.0001527; // !!TODO
   Config.gyro_gain_xneg     = 0.0001527; // !!TODO

   Config.gyro_gain_ypos     = 0.0001527; // !!TODO
   Config.gyro_gain_yneg     = 0.0001527; // !!TODO

   Config.gyro_gain_zpos     = 0.0001527; // !!TODO
   Config.gyro_gain_zneg     = 0.0001527; // !!TODO

   Config.cam_bias           =  0.0;      // !!TODO
   Config.cam_l_gain         = 15.0;      // !!TODO
   Config.cam_r_gain         = 15.0;      // !!TODO

   Config.battery_multiplier = 0.00984;

   Config.gyro_x_tc          = 0.0;       // !!TODO
   Config.gyro_y_tc          = 0.0;       // !!TODO
   Config.gyro_z_tc          = 0.0;       // !!TODO

//---------------------------------------------------------------------------------------------------------------------------------------------------
#else
#error CAMERA_ID
#endif
//---------------------------------------------------------------------------------------------------------------------------------------------------

   //
   // tuning parameters
   //
   
   Config.gyro_filter_k      = 3;
   Config.acc_filter_k       = 2;
   Config.servo_filter_k     = 0;
   
   Config.imu_rate_threshold = RAD(1.0);
   Config.imu_rate_duration  = 0.030; // 6 updates @ 200Hz (5ms) gyro data rate
   Config.imu_time_constant  = 0.5;
   }

// Show current settings.
//
void
CONFIG_dump()
   {
   printf("%u\n",              CAMERA_ID);
   
   printf("%d,%d,%d,\n",       Config.acc_x_bias, Config.acc_y_bias, Config.acc_z_bias);
   printf("%.7f,%.7f,%.7f,\n", Config.acc_x_gain, Config.acc_y_gain, Config.acc_z_gain);

   printf("%.7f,%.7f,\n",      Config.gyro_gain_xpos, Config.gyro_gain_xneg);
   printf("%.7f,%.7f,\n",      Config.gyro_gain_ypos, Config.gyro_gain_yneg);
   printf("%.7f,%.7f,\n",      Config.gyro_gain_zpos, Config.gyro_gain_zneg);

   printf("%.1f,%.1f,%.1f,\n", Config.cam_bias, Config.cam_l_gain, Config.cam_r_gain);
   printf("%.5f,\n",           Config.battery_multiplier);
   printf("%.1f,%.1f,%.1f,\n", Config.gyro_x_tc, Config.gyro_y_tc, Config.gyro_z_tc);
   
   printf("%u,%u,%u,\n",       Config.gyro_filter_k, Config.acc_filter_k, Config.servo_filter_k );
   printf("%.2f,%.3f,%.1f\n",  DEG(Config.imu_rate_threshold), Config.imu_rate_duration, Config.imu_time_constant);
   }
