// STmicroelectronics L3Gxxx 3-axis gyro.
//
// Gyro characteristics:
//
//    range        +/- 250 degrees-per-second
//    resolution   16 bits
//    sensitivity  .00875 degrees-per-second per digit
//    update rate  ~100/200/400/800 Hz

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

// Types.
//
typedef SWORD GYRO_DIGITS; // an unscaled gyro reading [range=-32k..+32k, nominal sensitivity=.00875 degrees-per-second per digit]

// Registers.
//
#define GYRO_CTRL_REG1  0x20
#define GYRO_CTRL_REG2  0x21
#define GYRO_CTRL_REG3  0x22
#define GYRO_CTRL_REG4  0x23
#define GYRO_CTRL_REG5  0x24
#define GYRO_REFERENCE  0x25
#define GYRO_OUT_TEMP   0x26
#define GYRO_STATUS_REG 0x27

#define GYRO_OUT_X_L    0x28
#define GYRO_OUT_X_H    0x29
#define GYRO_OUT_Y_L    0x2A
#define GYRO_OUT_Y_H    0x2B
#define GYRO_OUT_Z_L    0x2C
#define GYRO_OUT_Z_H    0x2D

#define GYRO_WHO_AM_I   0x0F

// Access.
//
//#define GYRO_ADDR 0x6b
#define GYRO_ADDR GYRO_TWI_ADDRESS

// Read gyro sensors, mapping sensor axes to body axes such that:
//    x points ahead (body roll axis)
//    y points right (body pitch axis)
//    z points down  (body yaw axis)
//    signs respect right hand rule
//
static void
GYRO_read_xyz(GYRO_DIGITS *x, GYRO_DIGITS *y, GYRO_DIGITS *z)
   {
   BYTE b[6];
   TWI_read_multi(GYRO_ADDR, GYRO_OUT_X_L | 0x80, sizeof(b), b); // 0x80 is autoincrement bit

   *z = -(b[0] | (b[1] << 8)); // X sensor
   *x = -(b[2] | (b[3] << 8)); // Y sensor
   *y =  (b[4] | (b[5] << 8)); // Z sensor
   }

#define GYRO_FILTER_K Config.gyro_filter_k

// Temperature compensation data.
//
static GYRO_DIGITS           GYRO_x_bias_ref; // reference zero-rate bias, set by GYRO_calibrate()
static GYRO_DIGITS           GYRO_y_bias_ref; // "
static GYRO_DIGITS           GYRO_z_bias_ref; // "
static SBYTE                 GYRO_t_ref;      // die temperature for which the above biases are valid

//------------------------------
// Interrupt communication area.
//
static volatile WORD         GYRO_hits;   // number of times data changed
static volatile WORD         GYRO_misses; // number of times data changed before we could read it (should only happen once, at startup)

static volatile GYRO_DIGITS  GYRO_x_bias; // zero-rate bias, temperature corrected every few seconds
static volatile GYRO_DIGITS  GYRO_y_bias; // "
static volatile GYRO_DIGITS  GYRO_z_bias; // "

static volatile GYRO_DIGITS  GYRO_x_urate; // gyro rate, bias corrected, updated at GYRO_HZ rate
static volatile GYRO_DIGITS  GYRO_y_urate; // "
static volatile GYRO_DIGITS  GYRO_z_urate; // "

static volatile GYRO_DIGITS  GYRO_x_srate; // gyro rate, bias corrected and smoothed, updated at GYRO_HZ rate
static volatile GYRO_DIGITS  GYRO_y_srate; // "
static volatile GYRO_DIGITS  GYRO_z_srate; // "

static volatile BOOL         GYRO_calibrating;
static volatile SDWORD       GYRO_x_sum;  // gyro calibration data, updated at GYRO_HZ rate
static volatile SDWORD       GYRO_y_sum;  // "
static volatile SDWORD       GYRO_z_sum;  // "
static volatile SWORD        GYRO_cnt;    // "
//
//------------------------------

// Update gyro rates.
// Called by interrupt.
//
static inline void
GYRO_update()
   {
   BYTE status = TWI_read(GYRO_ADDR, GYRO_STATUS_REG);

   if (status & 0x08)
      { // new data is available (note: timer tick interrupt fires at a rate faster than gyro data changes)
      
      // raw sensor readings
      //
      GYRO_DIGITS x, y, z;
      GYRO_read_xyz(&x, &y, &z);

      // accumulate data for zero rate bias calibration
      //
      if (GYRO_calibrating)
         {
         GYRO_x_sum += x;
         GYRO_y_sum += y;
         GYRO_z_sum += z;
         GYRO_cnt   += 1;
         return;
         }
      
      // remove zero rate biases
      //
      x -= GYRO_x_bias;
      y -= GYRO_y_bias;
      z -= GYRO_z_bias;
      
      // unsmoothed rates, for integrator
      //
      GYRO_x_urate = x;
      GYRO_y_urate = y;
      GYRO_z_urate = z;

      // smooth with a low pass filter
      // K = filter strength (0=none, 1=weak, 4+=strong)
      //
      const BYTE K = GYRO_FILTER_K;
      
      static SDWORD x_filter;
      x_filter = x_filter - (x_filter >> K) + x;
      GYRO_x_srate = x_filter >> K;

      static SDWORD y_filter;
      y_filter = y_filter - (y_filter >> K) + y;
      GYRO_y_srate = y_filter >> K;

      static SDWORD z_filter;
      z_filter = z_filter - (z_filter >> K) + z;
      GYRO_z_srate = z_filter >> K;

      GYRO_hits += 1;
      }

   if (status & 0x80)
      { // data changed before we had a chance to read it
      GYRO_misses += 1;
      }
   }

// --------------------------------------------------------------------
//                        Interface.
// --------------------------------------------------------------------

// Prepare gyros for use.
//
static void
GYRO_init()
   {
   //  data rate and bandwidth configuration settings
   //
   //  DR  BW    output data rate (Hz)   bandwidth/cutoff freq (Hz)
   //  --  --    ---------------------   --------------------------
   //               L3GD20  L3G4200D          L3GD20  L3G4200D
   //
   //  00  00           95  100                 12.5  same
   //  00  01           95  100                 25    same
   //  00  10           95  100                 25    same
   //  00  11           95  100                 25    same
   //
   //  01  00          190  200                 12.5  same
   //  01  01          190  200                 25    same
   //  01  10          190  200                 50    same
   //  01  11          190  200                 70    same
   //
   //  10  00          380  400                 20    same
   //  10  01          380  400                 25    same
   //  10  10          380  400                 50    same
   //  10  11          380  400                 100   110
   //
   //  11  00          760  800                 30    same
   //  11  01          760  800                 35    same
   //  11  10          760  800                 50    same
   //  11  11          760  800                 100   110

   #if GYRO_HZ != 200
   #error GYRO_HZ
   #endif

   // 200Hz data rate, 70Hz bandwidth, power on, enable all axes
   TWI_write(GYRO_ADDR, GYRO_CTRL_REG1, 0x7F); // 0111.1111

   // enable block (atomic) updates to LSB,MSB data pairs
   TWI_write(GYRO_ADDR, GYRO_CTRL_REG4, 0x80); // 1000.0000

#if 0
   BYTE who = TWI_read(GYRO_ADDR, GYRO_WHO_AM_I);
   printf("who=0x%02x\n", who); // expect 0xD3=L3G4200D (old chip) or 0xD4=L3GD20 (new chip)
   for (;;);
#endif

   // read sensors once to restart adc after changing settings
   GYRO_DIGITS x, y, z;
   while ((TWI_read(GYRO_ADDR, GYRO_STATUS_REG) & 8) == 0) ;
   GYRO_read_xyz(&x, &y, &z);
   }

// Fast blink led for N seconds during gyro calibration.
//
void
GYRO_blink(BYTE n)
   {
   LCD_text(1, 0);
   BYTE i;
   BYTE blinks = n * 8;
   BYTE boxes_drawn = 0;
   for (i = 0; i < blinks; ++i)
      {
      BYTE boxes_needed = i * LCD_COLS / blinks;
      for (; boxes_drawn < boxes_needed; ++boxes_drawn)
         LCD_box();
      TIME_pause(.125);
      LED_toggle();
      }
   LCD_end();
   }

// Get current die temperature.
// The returned value is 0 at about 95 degrees F and decreases by 1 digit per degree C.
//
SBYTE
GYRO_temperature()
   {
   DI();
   BYTE t = TWI_read(GYRO_ADDR, GYRO_OUT_TEMP);
   EI();
   return t;
   }

// Update zero rate biases to match current temperature.
//
void
GYRO_tc()
   {
   SBYTE t  = GYRO_temperature();
   SBYTE dT = t - GYRO_t_ref;
   
   GYRO_DIGITS x = GYRO_x_bias_ref + dT * Config.gyro_x_tc;
   GYRO_DIGITS y = GYRO_y_bias_ref + dT * Config.gyro_y_tc;
   GYRO_DIGITS z = GYRO_z_bias_ref + dT * Config.gyro_z_tc;

#if 0
   printf("tc\n");
   printf("t: %d -> %d\n", GYRO_t_ref,  t);
   printf("x: %d -> %d\n", GYRO_x_bias, x);
   printf("y: %d -> %d\n", GYRO_y_bias, y);
   printf("z: %d -> %d\n", GYRO_z_bias, z);
#endif
   
   GYRO_x_bias = x;
   GYRO_y_bias = y;
   GYRO_z_bias = z;
   }

// Read gyros for a few seconds and compute biases needed to zero the output rates.
// Assumption: device is motionless
//
void
GYRO_calibrate()
   {
   // accumulate data
   //
   GYRO_x_sum = 0;
   GYRO_y_sum = 0;
   GYRO_z_sum = 0;
   GYRO_cnt   = 0;
   GYRO_calibrating = 1;
   GYRO_blink(5); // 5 seconds == ~500 measurements @ 100Hz
   GYRO_calibrating = 0;
   
   // calculate new biases
   //
   GYRO_x_bias = GYRO_x_sum / GYRO_cnt;
   GYRO_y_bias = GYRO_y_sum / GYRO_cnt;
   GYRO_z_bias = GYRO_z_sum / GYRO_cnt;

   // remember them for use in temperature correction
   //
   GYRO_x_bias_ref = GYRO_x_bias;
   GYRO_y_bias_ref = GYRO_y_bias;
   GYRO_z_bias_ref = GYRO_z_bias;
   GYRO_t_ref      = GYRO_temperature();
   }

// Calculate how far gyros have turned during current imu timestep, in radians.
// Note: we use unsmoothed gyro rates to minimize imu lag (any jitter will get averaged out by imu integrator)
//
void
GYRO_getRotations(FLOAT *xp, FLOAT *yp, FLOAT *zp)
   {
   DI();
   GYRO_DIGITS x = GYRO_x_urate;
   GYRO_DIGITS y = GYRO_y_urate;
   GYRO_DIGITS z = GYRO_z_urate;
   EI();

   const FLOAT IMU_TIMESTEP = (1.0 / IMU_HZ);
   *xp = x * (x > 0 ? Config.gyro_gain_xpos : Config.gyro_gain_xneg) * IMU_TIMESTEP;
   *yp = y * (y > 0 ? Config.gyro_gain_ypos : Config.gyro_gain_yneg) * IMU_TIMESTEP;
   *zp = z * (z > 0 ? Config.gyro_gain_zpos : Config.gyro_gain_zneg) * IMU_TIMESTEP;
   }

// Get (smoothed) gyro rates, in radians/sec.
//
FLOAT
GYRO_getRollRate()
   {
   DI();
   GYRO_DIGITS x = GYRO_x_srate;
   EI();
   return x * (x > 0 ? Config.gyro_gain_xpos : Config.gyro_gain_xneg);
   }

FLOAT
GYRO_getPitchRate()
   {
   DI();
   GYRO_DIGITS y = GYRO_y_srate;
   EI();
   return y * (y > 0 ? Config.gyro_gain_ypos : Config.gyro_gain_yneg);
   }

FLOAT
GYRO_getYawRate()
   {
   DI();
   GYRO_DIGITS z = GYRO_z_srate;
   EI();
   return z * (z > 0 ? Config.gyro_gain_zpos : Config.gyro_gain_zneg);
   }
