// Track a motorcycle's orientation using gyros (and, possibly, accelerometers).
// Most of the algorithms here were developed by Bill Premerlani. The motorcycle-specific drift correction ideas are mine.
// 09 Nov 2012 Derek Lieber
//
// References:
// [Art 1] "Direction Cosine Matrix IMU Theory"                            http://gentlenav.googlecode.com/files/DCMDraft2.pdf
// [Art 2] "Computing Euler Angles From Direction Cosines"                 http://gentlenav.googlecode.com/files/EulerAngles.pdf
// [Art 3] "A Sensor Fusion Method for Smart phone Orientation Estimation" http://www.cms.livjm.ac.uk/pgnet2012/Proceedings/Papers/1569603133.pdf
// [Art 4] "Tilt Sensing Using Linear Accelerometers"                      http://cache.freescale.com/files/sensors/doc/app_note/AN3461.pdf
//
// We use the "aerospace convention" for rotations throughout.
// That is:
// - x is roll  axis, points ahead, positive rotation == roll right
// - y is pitch axis, points right, positive rotation == pitch up
// - z is yaw   axis, points down,  positive rotation == yaw right
// Ref: [Art 2, Fig 2]
//

//---------------------------------------------------------------------------------------------------------------------------------------------------
// Interrupt communication area - updated by interrupt at a rate of IMU_HZ.
//                                                                                                                                                
// Current orientation of bike and ground with respect each other, as a rotation matrix.                                                          
// Columns are projections of bike xyz axes on ground xyz axes.                                                                                   
// Rows    are projections of ground xyz axes on bike xyz axes.                                                                                   
//                                                                                                                                                
static volatile FLOAT                                                                                                                             
   Rxx, Rxy, Rxz,                                                                                                                                 
   Ryx, Ryy, Ryz,                                                                                                                                 
   Rzx, Rzy, Rzz;                                                                                                                                 
                                                                                                                                                  
// Estimated "drift" error awaiting correction, in radians.                                                                                       
//                                                                                                                                                
static volatile FLOAT IMU_rollError, IMU_pitchError, IMU_yawError;                                                                                
                                                                                                                                                  
// Have all the above values been set (ie. by IMU_align)?                                                                                         
//                                                                                                                                                
static volatile BOOL  IMU_aligned;                                                                                                                
                                                                                                                                                  
// Debugging stuff.                                                                                                                               
//                                                                                                                                                
static volatile WORD  IMU_hits;                      // number of times imu has executed
static volatile BOOL  IMU_upright;                   // does imu think bike is currently straight and level?
static volatile BOOL  IMU_drift_correction_disabled; // turn off drift correction
static volatile BOOL  IMU_debug_snapshots_enabled;   // turn on debug data snapshots for workbench
static volatile FLOAT IMU_rollDelta,  IMU_rollCorr;  // debug...
static volatile FLOAT IMU_pitchDelta, IMU_pitchCorr; // ...data
static volatile FLOAT IMU_yawDelta,   IMU_yawCorr;   // ...snapshots
//                                                                                                                                                
//---------------------------------------------------------------------------------------------------------------------------------------------------

// Initialize the orientation matrix.
// Taken:   bike's orientation with respect to ground, in radians
// Updated: Rxx..Rzz
// Ref: [Art 2, Eqn 2] and [http://en.wikipedia.org/wiki/Rotation_matrix]
//
void
IMU_set(FLOAT roll, FLOAT pitch, FLOAT yaw)
   {
   FLOAT
  
   R = roll,
   P = pitch,
   Y = yaw,
   
   sinR = sin(R), cosR = cos(R),
   sinP = sin(P), cosP = cos(P),
   sinY = sin(Y), cosY = cos(Y);

   // The full rotation matrix, R, is formed by applying individual axis rotations in yaw, pitch, roll order.
   // We do this by premultiplying, right to left:
   //    R = Rr * Rp * Ry
   // Where:
   //         1    0     0          cosR  0 sinR          cosY -sinY 0
   //    Rr = 0 cosP -sinP     Rp =    0  1    0    Ry =  sinY  cosY 0
   //         0 sinP  cosP         -sinR  0 cosR             0    0  1

   DI();
   
   Rxx = cosP * cosY; Rxy = sinR * sinP * cosY - cosR * sinY; Rxz = cosR * sinP * cosY + sinR * sinY;
   Ryx = cosP * sinY; Ryy = sinY * sinP * sinY + cosR * cosY; Ryz = cosR * sinP * sinY - sinR * cosY;
   Rzx = -sinP;       Rzy = sinR * cosP;                      Rzz = cosR * cosP;

   IMU_rollError  = 0;
   IMU_pitchError = 0;
   IMU_yawError   = 0;
   
   if (IMU_debug_snapshots_enabled) IMU_rollDelta = IMU_rollCorr = IMU_pitchDelta = IMU_pitchCorr = IMU_yawDelta = IMU_yawCorr = 0;

   EI();
   }

// Get the bike's gyro-integration-estimated orientation with respect to ground, in radians.
// Ref: [Art 2, Eqn 3]
//
static FLOAT
IMU_getRollAngle()
   { 
   DI();
   FLOAT a = Rzy;
   FLOAT b = Rzz;
   EI();
   return atan2(a, b);
   }

static FLOAT
IMU_getPitchAngle()
   {
   DI();
   FLOAT a = Rzx;
   EI();
   return -asin(a);
   } 

static FLOAT
IMU_getYawAngle()
   { 
   DI();
   FLOAT a = Ryx;
   FLOAT b = Rxx;
   EI();
   return atan2(a, b);
   }

// Align orientation matrix, bike, and gyros with each other.
// The best way to ensure proper alignment is:
// - bike is standing still                <- gyro rates == 0
// - bike is upright on level ground       <- bike roll == 0, pitch == 0
// - camera is pointed straight ahead      <- gyro yaw == 0
// - x,y accelerometers are reading zero   <- gyro roll == 0, pitch == 0
//
static void
IMU_align()
   { 
   IMU_aligned = 0;
   GYRO_calibrate();
   IMU_set(0, 0, 0);
   IMU_aligned = 1;  
   }

// Rotate orientation matrix to follow bike's motion.
// Called by interrupt.
//
// Taken:   differential rotations of bike around its gyro axes, in radians
// Updated: Rxx..Rzz
//
// This algorithm is neat because it contains no divisions, trig functions, or square roots.
//
// I'm somewhat puzzled about the dimensions in use here.
// We seem to be intermixing dimensionless units (the matrix elements) with angular units (the applied differential rotations, in radians).
// The same thing happens later when we compute the drift corrections: we intermix matrix elements with quasi-normalized accelerometer components.
// I'm not sure why this all works out. I guess it's all a bunch of approximations that get cleaned up by the re-orthonormalization step.
//
static inline void
IMU_rotate(FLOAT dx, FLOAT dy, FLOAT dz)
   {
   // The update formula is:   
   //
   //    R(t + dt) = R(t) * Q    
   //
   // where:
   //            Rxx Rxy Rxz
   //       R =  Ryx Ryy Ryz
   //            Rzx Rzy Rzz
   // and:
   //             1 -dz  dy
   //       Q  =  dz  1 -dx
   //            -dy dx   1
   //
   // R is orientation matrix.
   // Q is differential rotation measured in bike frame of reference.
   //
   // dx = rotation about bike's roll  axis (points ahead), positive = roll right
   // dy = rotation about bike's pitch axis (points right), positive = pitch up
   // dz = rotation about bike's yaw   axis (points down),  positive = yaw right
   //
   // [Art 1, Eqn 17]
   //

   FLOAT
   Qxx =   1,  Qxy = -dz,  Qxz =  dy,
   Qyx =  dz,  Qyy =   1,  Qyz = -dx,
   Qzx = -dy,  Qzy =  dx,  Qzz =   1;
   
   FLOAT
   Txx = Rxx * Qxx + Rxy * Qyx + Rxz * Qzx,  Txy = Rxx * Qxy + Rxy * Qyy + Rxz * Qzy,  Txz = Rxx * Qxz + Rxy * Qyz + Rxz * Qzz,
   Tyx = Ryx * Qxx + Ryy * Qyx + Ryz * Qzx,  Tyy = Ryx * Qxy + Ryy * Qyy + Ryz * Qzy,  Tyz = Ryx * Qxz + Ryy * Qyz + Ryz * Qzz,
   Tzx = Rzx * Qxx + Rzy * Qyx + Rzz * Qzx,  Tzy = Rzx * Qxy + Rzy * Qyy + Rzz * Qzy,  Tzz = Rzx * Qxz + Rzy * Qyz + Rzz * Qzz;

   Rxx = Txx;  Rxy = Txy;  Rxz = Txz;
   Ryx = Tyx;  Ryy = Tyy;  Ryz = Tyz;
   Rzx = Tzx;  Rzy = Tzy;  Rzz = Tzz;

   // Re-orthonormalize the matrix with the following objectives:
   // The dot product of the X & Y rows should be zero.
   // The Z row should be equal to the cross product of the X & Y rows.
   // Each row should be of unit magnitude.
   //
   // The idea is that the three rows and columns will always be approximately perpendicular, because we are going to maintain them that
   // way, and we are going to maintain their lengths to be one, but we will need to fix up slight rotational errors.
   // For example, suppose vectors A and B are almost, but not exactly perpendicular, and we want to adjust them to make them closer to perpendicular.
   // We do not want to change their magnitude, we just want to rotate them. That means the adjustment to each of them is perpendicular.
   // Since B is perpendicular to A, when we want to rotate A a little bit, we simply add a portion of B. And vice-versa when we want to rotate B.
   //
   // So we take the dot product of the X and Y rows to find out if they are perpendicular. If they are, the dot product will be zero.
   // If they are not, the dot product will be measure of how much they need to be rotated toward or away from each other to be perpendicular.
   // Since we have no way of knowing whether X or Y are more likely to be correct, we split the difference, and adjust both X and Y by half.
   
   // Measure how much rows X and Y are rotated towards each other [Art 1, Eqn 18].
   //
   FLOAT dot = Rxx * Ryx + Rxy * Ryy + Rxz * Ryz;
   
   // Now rotate each away from the other by half that amount, thereby restoring their orthogonality [Art 1, Eqn 19].
   //
   FLOAT half = .5 * dot;

   // temporary copy of row X
   Txx =  Rxx;
   Txy =  Rxy;
   Txz =  Rxz;

   // rotate row X away from Y
   Rxx -= half * Ryx;
   Rxy -= half * Ryy;
   Rxz -= half * Ryz;

   // rotate row Y away from X
   Ryx -= half * Txx;
   Ryy -= half * Txy;
   Ryz -= half * Txz;

   // Set row Z to cross product of X and Y rows [Art 1, Eqn 20].
   //
   Rzx = Rxy * Ryz - Rxz * Ryy;
   Rzy = Rxz * Ryx - Rxx * Ryz;
   Rzz = Rxx * Ryy - Rxy * Ryx;

   // Normalize each row by dividing each element by the row's magnitude (square root of sum of squares)
   // noting that, since the magnitude should be approximately 1, we can use a short Taylor expansion
   // to compute the reciprocal square root:    1/sqrt(x) ~=  1/2 * (3 - x) for x ~= 1
   //
   FLOAT nx = .5 * (3 - (Rxx * Rxx + Rxy * Rxy + Rxz * Rxz));
   FLOAT ny = .5 * (3 - (Ryx * Ryx + Ryy * Ryy + Ryz * Ryz));
   FLOAT nz = .5 * (3 - (Rzx * Rzx + Rzy * Rzy + Rzz * Rzz));

   Rxx *= nx; Rxy *= nx; Rxz *= nx;
   Ryx *= ny; Ryy *= ny; Ryz *= ny;
   Rzx *= nz; Rzy *= nz; Rzz *= nz;
   }

// Update orientation matrix in step with bike's motions and apply drift corrections.
// Called by interrupt at a rate of IMU_HZ.
//
static inline void
IMU_update()
   {
   if (!IMU_aligned)
      return;

   IMU_hits += 1;
      
   // Estimate how much the bike has rotated during this timestep.
   // Note that these rotations are with respect to the bike reference frame, not the ground.
   //
   FLOAT rollDelta, pitchDelta, yawDelta;
   GYRO_getRotations(&rollDelta, &pitchDelta, &yawDelta);

   // Estimate a correction that will counteract any accumulated errors in the orientation matrix.
   //
   #define IMU_RATE_THRESHOLD Config.imu_rate_threshold // example value: 1deg/sec
   #define IMU_RATE_DURATION  Config.imu_rate_duration  // example value: .03sec
   #define IMU_TIME_CONSTANT  Config.imu_time_constant  // example value: .5sec

   // Guess the bike's stance by examining yaw rate:
   //    yaw-rate==0 roll-rate!=0 --> bike is upright, at an inflection point in a transition from left-to-right or right-to-left turn
   //    yaw-rate==0 roll-rate==0 --> bike is upright, on a straightaway
   // Note that yaw and roll rates provide no way to guess the bike's pitch orientation.
   //
   BOOL upright = fabs(GYRO_getYawRate()) <= IMU_RATE_THRESHOLD;
   
   // Avoid false indications induced by noise or vibration - stance must persist before it's considered valid.
   //
   static FLOAT duration;
   if (upright)
      {
      duration += 1.0 / IMU_HZ;
      if (duration < IMU_RATE_DURATION) 
         upright = 0; // rate hasn't persisted long enough yet
      }
   else
      duration = 0;   // rate isn't low enough yet
   
   if (upright)
      {
      // Take a drift error "snapshot".
      // Ignore accelerometers and assume that the bike has momentairly returned to a (0,0,0) orientation with respect to ground.
      // If it's been tracking correctly, the current matrix-indicated bike orientation with respect to ground should be nearly the same,
      // so we can use small angle approximations to measure any differences and use them as drift error measurements.
      // Furthermore, since the bike and drift correction reference frames are (nearly) aligned, we can later use
      // simple addition to combine bike rotations and drift corrections into a single rotation update.
      // Caveats: cross-axis effects will be introduced if current pitch angle != 0 (for example on a hill)
      // or if startup pitch/roll angles != 0 (for example bike was sitting on sidestand or on an incline).
      //
      IMU_rollError  =  Rzy; // (smax)
      IMU_pitchError = -Rzx; // (smax)
      IMU_yawError   =  0;
      }

   // Update stance indicator.
   //
   IMU_upright = upright;
   LED_set(upright);

   // Apply error correction at rate specified by desired time constant.
   //
   const FLOAT T = IMU_TIME_CONSTANT; // time constant, in seconds
   const FLOAT C = IMU_HZ;            // number of corrections to apply per second
   const FLOAT K = 1.0 / T / C;       // fraction to apply per correction

   FLOAT rollCorr  = - K * IMU_rollError;
   FLOAT pitchCorr = - K * IMU_pitchError;
   FLOAT yawCorr   = - K * IMU_yawError;

   // Save remaining error. We'll correct more of it at next timestep (until overwritten by a new starting estimate).
   //
   IMU_rollError  += rollCorr;
   IMU_pitchError += pitchCorr;
   IMU_yawError   += yawCorr;
   
   // Caveat: The way we're implementing this scheme to "snapshot" an error and then spread out the correction over multiple timesteps is not quite
   // correct in all cases. In particular, if the bike enters a turn while we're still applying residual corrections (that were originally measured 
   // when bike and ground frame were nearly aligned) they'll be incorrect for the new, now rotated, bike frame. This will lead to various cross-axis
   // effects in the camera tracking.
   //
   // Cases involving a single-axis error and subsequent rotation about that same axis during the correction period will work fine, but the
   // right way to handle general multi-axis corrections would be to transform them back and forth between ground- and bike- reference
   // frames at each timestep. In practice, however, this is unnecessary unless the bike were to climb or descend while rolling into a turn.
   //
   // For now, we'll simply try to minimize such errors by keeping the correction period short (small IMU_TIME_CONSTANT),
   // but still long enough that the camera smoothly blends in corrections without abrupt movements.
   //

   // Keep orientation matrix purely gyro-based during sensitivity measurements on workbench.
   //
   if (IMU_drift_correction_disabled)
      rollCorr = pitchCorr = yawCorr = 0;

   // Take a data snapshot for debugging.
   //
   if (IMU_debug_snapshots_enabled)
      {
      IMU_rollDelta  = rollDelta;
      IMU_rollCorr   = rollCorr;
      
      IMU_pitchDelta = pitchDelta;
      IMU_pitchCorr  = pitchCorr;
      
      IMU_yawDelta   = yawDelta;
      IMU_yawCorr    = yawCorr;
      }

   // Apply bike rotations and drift corrections to orientation matrix. We assume that the bike rotations and drift rotations are small and in
   // the same frame of reference, so they may be combined using simple addition. See, however, the above caveat.
   //
   IMU_rotate(rollDelta + rollCorr, pitchDelta + pitchCorr, yawDelta + yawCorr);
   }

// --------------------------------------------------------------------------------------------------------------------------------------------------
// Side note - calculating rotation angles between two sets of orthonormal coordinate axes using small angle approximations (smax).
// --------------------------------------------------------------------------------------------------------------------------------------------------
//
// The following expressions are exact but computationally expensive, requiring 3 atan's, 1 asin, 1 sqrt:
//    roll angle difference (about x-axis) = IMU_getRollAngle()  - ACC_getRollAngle()
//   pitch angle difference (about y-axis) = IMU_getPitchAngle() - ACC_getPitchAngle()
//
// If the two sets of coordinate axes are nearly aligned, the following small angle approximations are cheaper and nearly as accurate:
//    roll angle difference (about x-axis), in radians ~= difference in y components
//   pitch angle difference (about y-axis), in radians ~= difference in x components
//
// That is:
//    roll angle difference (about x-axis), in radians ~=   Rzy - ACC_getY()   <- matrix components and accelerometer values normalized
//   pitch angle difference (about y-axis), in radians ~= -(Rzx - ACC_getX())  <- ...to same range (-1 .. +1)
//
// If the bike is vertical, then the x and y accerometers read zero, and this redues to:
//    roll angle difference (about x-axis), in radians ~=   Rzy
//   pitch angle difference (about y-axis), in radians ~= - Rzx
//
// Even if the small angle assumption is violated, we can use these approximations to get a general sense of the differences.
//
// --------------------------------------------------------------------------------------------------------------------------------------------------
// Side note - time constant for slowly applied corrections.
// --------------------------------------------------------------------------------------------------------------------------------------------------
//
// Example:   T = 5 seconds
//            C = 50 corrections per second
//            K = 1/T/C = .004 = .4% per correction
// Check:     50 corrections per second * .4% per correction * 5 seconds = 100%
//            (Actually, the time constant tells how long to reach 1/e (=36%) of the initial value, not 100%).
//
// --------------------------------------------------------------------------------------------------------------------------------------------------
// Side note - cross-axis coupling due to imu misalignment.
// --------------------------------------------------------------------------------------------------------------------------------------------------
//
// Misalignment between the matrix axes, those of the bike, and those of the gyros (either due to setup errors or to erroneously applied drift
// corrections) will produce puzzling camera effects. When the axes are properly aligned, the camera will turn in response to roll motions only.
// However, when the axes are misaligned, the camera will begin to turn also in response to yaw. For example, if the bike is executing a constant
// radius turn, the camera will keep rotating even though the roll rate is zero. This will continue through 180 degrees of turn and then reverse
// direction during the second 180 degrees. Or if the bike is executing a sequence of left/right turns, the camera will overcorrect first one way, 
// then the other, generating a peculiar rocking motion in the image. The only way to avoid this kind of behavior is to ensure that the angles
// used to initialize the matrix and to apply subsequent drift corrections are chosen reliably. For example, if accelerometers are used to determine
// orientation angles, then centrifugal accelerations must be accounted for (or the bike must be in a state of motion where centrifugal accelerations
// are zero, ie. either at rest or at constant speed on a straightaway). Accelerometer noise due to engine vibration can also introduce unwanted 
// cross-axis coupling effects. This can be mitigated by choosing enough smoothing (ACC_FILTER_K) to reduce jitter but not so much that we
// introduce excessive lag in the angle readouts. Ultimately, I decided that using accelerometers was impossible due to their vibration sensitivity.
// Using purely gyro-based heuristics worked out much better.
//
// --------------------------------------------------------------------------------------------------------------------------------------------------
