// Call a function periodically via hardware timer interrupt.
//
// Units:      TIMER0 channel A
// Interrupts: TIMER0_COMPA_vect
//
// Usage:      #define TICKER_HZ 1000
//        /or/ #define TICKER_HZ  400
//        /or/ #define TICKER_HZ  200
//             #include "ticker.h"

#ifdef TICKS_PER_SEC
#error TICKS_PER_SEC is obsolete - use TICKER_HZ instead
#endif

// Timer interrupt callback function.
//
typedef void (*TICKER_FUNC)();
static TICKER_FUNC TICKER_func;

// Interrupt service routine for TIMER0.
//
ISR(TIMER0_COMPA_vect)
   {
   (*TICKER_func)();
   }

// Set TIMER0 to call specified function at a rate of TICKER_HZ.
// Taken:    function to be called
// Returned: nothing
//
// We use CTC ("clear on terminal count") wave generation mode.
// The counter continually runs from 0 to TOP and then resets to 0.
// An interrupt is generated each time TOP is reached.
//
// The TOP count needed for a given interrupt rate is:
//    TOP = (FREQ / PRESCALER / RATE) - 1             
//
// Conversely the interrupt rate is: 
//    RATE = FREQ / PRESCALER / (TOP + 1)
//
// For FREQ =  8Mhz, PRESCALER = 64, RATE = 1000Hz:
//    TOP = (8000000 / 64 / 1000) - 1 = 124           [ actual rate = 8000000 /  64 / (124 + 1) = 1000  ]
//
// For FREQ =  8Mhz, PRESCALER = 256, RATE = 400Hz:
//    TOP = (8000000 / 256 / 400) - 1 = 77.125 => 77  [ actual rate = 8000000 / 256 / (77 + 1)  = 400.6 ]
//
// For FREQ =  8Mhz, PRESCALER = 256, RATE = 200Hz:
//    TOP = (8000000 / 256 / 200) - 1 = 155.25 => 155 [ actual rate = 8000000 / 256 / (155 + 1) = 200.3 ]
//
static void
TICKER_init(TICKER_FUNC f)
   {
   TICKER_func = f;

   // set timer for "clear on terminal count" operation (mode 2)
   //
   TCCR0A = 0
          | (0 << WGM00) // CTC wave generation (mode 2)
          | (1 << WGM01) // CTC wave generation (mode 2)
          ;

   TCCR0B = 0
          | (0 << WGM02) // CTC wave generation (mode 2)
          ;

   // set terminal count value (TOP) at which to generate a
   // "compare match A" interrupt and clear the counter back to zero
   // note that TIMER0 is an 8 bit timer, so TOP must be <= 255
   //
#if CLOCK_MHZ != 8
#error CLOCK_MHZ
#endif

#if   TICKER_HZ == 1000
   OCR0A = 124;
#elif TICKER_HZ == 400
   OCR0A =  77;
#elif TICKER_HZ == 200
   OCR0A = 155;
#else
#error TICKER_HZ
#endif
   
   // enable "TIMER0 compare match A" interrupts
   //
   TIMSK0 |= (1 << OCIE0A);

   // start the timer by enabling the prescaler
   //
   TCCR0B = 0
#if   TICKER_HZ == 1000
          | (1 << CS00)  // timer clock = system clock / 64
          | (1 << CS01)  // "
          | (0 << CS02)  // "
#elif TICKER_HZ == 400
          | (0 << CS00)  // timer clock = system clock /  256
          | (0 << CS01)  // "
          | (1 << CS02)  // "
#elif TICKER_HZ == 200
          | (0 << CS00)  // timer clock = system clock /  256
          | (0 << CS01)  // "
          | (1 << CS02)  // "
#else
#error TICKER_HZ
#endif
          ;
   }
