// Talk to devices via TWI (two-wire interface).
//
// Units:      TWI
// Interrupts: none
// Pins:       SDA (PORTC4)
//             SCL (PORTC5)
// Clock:      8Mhz
//
// Usage:      #define TWI_KHZ  50
//        /or/ #define TWI_KHZ 100
//             #include "twi.h"
//
// 16 Oct 2011 Derek Lieber
//

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

#include <util/twi.h> // twi status codes - /usr/lib/avr/include/util/twi.h

// TWI fault notifier.
//
typedef void (*TWI_FUNC)();
static TWI_FUNC TWI_fault;

// Execute a TWI operation.
// Taken:    TWCR command bits
// Returned: TWSR status bits
//
static BYTE
TWI_execute(BYTE op)
   {
   // initiate
   TWCR = op;
   
   // wait for completion
   while (!(TWCR & (1 << TWINT)));
   
   // get status bits
   return TWSR & ~((1 << TWPS1) | (1 << TWPS0));
   }

// Handle a TWI error.
// Taken:    operation type (start, send, recv, stop)
//           reported TWI status
//           expected TWI status
// Returned: does not return
//
static void 
TWI_error(const char *op, BYTE status, BYTE expected)
   {
 //printf("twi error: op=%s status=0x%02x expected=0x%02x\n", op, status, expected);
   for (;;) TWI_fault();
   }

// Start a TWI transaction.
//
static void
TWI_start()
   {
   BYTE status = TWI_execute((1 << TWINT) | (1 << TWEN) | (1 << TWSTA));
   if (status != TW_START && status != TW_REP_START)
      TWI_error("start", status, TW_START);
   }

// Send a byte to TWI slave.
// Taken:    byte to send
//           status expected after operation completes
// Returned: nothing (never returns if failure)
//
static void
TWI_send(BYTE data, BYTE expected)
   {
   TWDR = data;
   BYTE status = TWI_execute((1 << TWINT) | (1 << TWEN));
   if (status != expected)
      TWI_error("send", status, expected);
   }

// Receive a byte from TWI slave.
// Taken:    should master acknowledge reception?
// Returned: received data (never returns if failure)
//
static BYTE
TWI_recv(BOOL ack)
   {
   if (ack)
      {
      BYTE status = TWI_execute((1 << TWINT) | (1 << TWEN) | (1 << TWEA));
      if (status != TW_MR_DATA_ACK)
         TWI_error("recv", status, TW_MR_DATA_ACK);
      }
   else
      {
      BYTE status = TWI_execute((1 << TWINT) | (1 << TWEN));
      if (status != TW_MR_DATA_NACK)
         TWI_error("recv", status, TW_MR_DATA_NACK);
      }
   
   return TWDR;
   }

// Finish a TWI transaction.
//
static void
TWI_stop()
   {
   TWCR = ((1 << TWINT) | (1 << TWEN) | (1 << TWSTO));
   // no wait, no status
   }
   
// --------------------------------------------------------------------
//                          Interface.
// --------------------------------------------------------------------

// Prepare TWI for use.
// Taken:    TWI fault notification function
// Returned: nothing
//
static void
TWI_init(TWI_FUNC fault)
   {
   TWI_fault = fault;
   
#if CLOCK_MHZ != 8
#error CLOCK_MHZ
#endif

#if TWI_KHZ == 50
   // set TWI clock frequency to 50Khz
   //
   // CLOCK_HZ      = 8,000,000
   // TWBR          = 72
   // prescaler     = 1
   // SCL frequency = CLOCK_HZ  / (16 + 2 * TWBR * prescaler)
   //               = 8,000,000 / (16 + 2 *  72  * 1)
   //               = 50KHz

   TWSR = (0 << TWPS1) | (0 << TWPS0); // prescaler = 1
   TWBR = 72;                          // divisor
   
#elif TWI_KHZ == 100
   
   // set TWI clock frequency to 100Khz
   //
   // CLOCK_HZ      = 8,000,000
   // TWBR          = 32
   // prescaler     = 1
   // SCL frequency = CLOCK_HZ  / (16 + 2 * TWBR * prescaler)
   //               = 8,000,000 / (16 + 2 *  32  * 1)
   //               = 100KHz

   TWSR = (0 << TWPS1) | (0 << TWPS0); // prescaler = 1
   TWBR = 32;                          // divisor
   
#else
#error TWI_KHZ
#endif
   }
   
// Write one byte to a TWI device.
// Taken:    7 bit device address
//           8 bit register number
//           8 bit value to be written
// Returned: nothing (no return if operation fails)
//
void
TWI_write(WORD device_address, BYTE register_number, BYTE value)
   {
   // begin transaction (ST)
   TWI_start();

   // send slave address (SLA+W)
   TWI_send((device_address << 1) | TW_WRITE, TW_MT_SLA_ACK);

   // send register number (SUB) 
   TWI_send(register_number, TW_MT_DATA_ACK);
   
   // send register value (DATA)
   TWI_send(value, TW_MT_DATA_ACK);

   // end transaction (SP)
   TWI_stop();
   }

// Read one byte from a TWI device.
// Taken:    7 bit device address
//           8 bit register number
// Returned: value read (no return if operation fails)
//
BYTE
TWI_read(WORD device_address, BYTE register_number)
   {
   // begin transaction (ST)
   TWI_start();

   // send slave address (SLA+W)
   TWI_send((device_address << 1) | TW_WRITE, TW_MT_SLA_ACK);

   // send register number (SUB) 
   TWI_send(register_number, TW_MT_DATA_ACK);

   // begin transaction (SR)
   TWI_start();

   // send slave address (SLA+R)
   TWI_send((device_address << 1) | TW_READ, TW_MR_SLA_ACK);

   // receive register value (DATA)
   BYTE value = TWI_recv(0); // nack last (only) byte

   // end transaction (SP)
   TWI_stop();

   return value;
   }

// Read multiple bytes from a TWI device.
// Taken:    7 bit device address
//           8 bit register number
//           number of bytes to read
//           place to put them
// Returned: nothing (no return if operation fails)
//
void
TWI_read_multi(WORD device_address, BYTE register_number, BYTE n, BYTE *dst)
   {
   // begin transaction (ST)
   TWI_start();

   // send slave address (SLA+W)
   TWI_send((device_address << 1) | TW_WRITE, TW_MT_SLA_ACK);

   // send register number (SUB) 
   TWI_send(register_number, TW_MT_DATA_ACK);

   // begin transaction (SR)
   TWI_start();

   // send slave address (SLA+R)
   TWI_send((device_address << 1) | TW_READ, TW_MR_SLA_ACK);

   // receive register values (DATA)
   while (n--)
       *dst++ = TWI_recv(n != 0); // ack all bytes but last

   // end transaction (SP)
   TWI_stop();
   }
