// A motorcycle gyrocam controller.
// 15 Dec 2012 Derek Lieber
//
// Hardware: 
//    Atmel ATMEGA328P mcu running at 8Mhz on 5 volts
//    Pololu L3G4200D     3-axis gyro
// or Pololu L3GD20       3-axis gyro
// or Pololu MiniIMU-9 V2 3-axis gyro, accelerometer, magnetometer
//    Hitec HS-425BB servo
//    Contour Roam camera with lens barrel dust seal & clickers removed
//
// I2C addresses:
//   gyro          = 110.1001 = 69 for L3G4200D
//                 = 110.1011 = 6B for L3GD20 or MiniIMU
//   accelerometer = 001.1001 = 19
//   magnetometer  = 001.1110 = 1E
//
// History: 17 May 2012 - tilt mount, single axis gyro
//          07 Jul 2012 - pulley mount
//          30 Jul 2012 - Jared's video
//          15 Aug 2012 - 1:1 gear mount
//          01 Sep 2012 - 1.5:1 gear mount, three axis gyro, single axis integrator, single shot error correction
//          05 Oct 2012 - Richard Harris' video
//          08 Nov 2012 - three axis gyro, three axis integrator, blended error correction
//          15 Dec 2012 - first completely successfull gap runs
//
//----------------------------------------------------------------------
// Atmega328 pin and port allocations.
//----------------------------------------------------------------------
//
//                        AREF = pin 21
//                        AVCC = pin 20 <- 5v
//                         VCC = pin  7 <- 5v
//                        AGND = pin 22 <- 0v
//                         GND = pin  8 <- 0v
//                      #RESET = pin  1 <- 10k pullup + grounding button
//
// [ICP1][PCINT0][CLKO] PORTB0 = pin 14 
// [OC1A][PCINT1]       PORTB1 = pin 15 -> servo
// [OC1B][PCINT2] [#SS ]PORTB2 = pin 16
// [OC2A][PCINT3] [MOSI]PORTB3 = pin 17
//       [PCINT4] [MISO]PORTB4 = pin 18
//       [PCINT5] [SCK ]PORTB5 = pin 19
//       [PCINT6][XTAL1]PORTB6 = pin  9
//       [PCINT7][XTAL2]PORTB7 = pin 10
//
//       [PCINT8] [ADC0]PORTC0 = pin 23 -> status led
//       [PCINT9] [ADC1]PORTC1 = pin 24 <- battery voltage divider
//       [PCINT10][ADC2]PORTC2 = pin 25 -> power switch
//       [PCINT11][ADC3]PORTC3 = pin 26 
// [SDA] [PCINT12][ADC4]PORTC4 = pin 27 <-> imu SDA
// [SCL] [PCINT13][ADC5]PORTC5 = pin 28 ->  imu SCL
//                      PORTC6 = n/a
//                      PORTC7 = n/a
//
//       [PCINT16] [RXD]PORTD0 = pin  2 <- serial connection + 10k pullup
//       [PCINT17] [TXD]PORTD1 = pin  3 -> serial connection
//       [PCINT18][INT0]PORTD2 = pin  4 
// [OC2B][PCINT19][INT1]PORTD3 = pin  5 
// [XCK] [PCINT20]  [T0]PORTD4 = pin  6 
// [OC0B][PCINT21]  [T1]PORTD5 = pin 11 
// [OC0A][PCINT22][AIN0]PORTD6 = pin 12 
//       [PCINT23][AIN1]PORTD7 = pin 13

#define CLOCK_INTERNAL 1           // internal oscillator
#define CLOCK_MHZ      8           // 8 Mhz system clock

// Timebase constants and sensor data rates.
//
#define TICKER_HZ 400              // timer tick interrupt rate
#define ACC_HZ     50              // accelerometer data rate
#define GYRO_HZ   200              // gyro data rate
#define IMU_HZ    200              // imu update rate
#define CAM_HZ    200              // camera update rate
#define TWI_KHZ   100              // twi clock rate

#include <avr/io.h>                // avr architecture - see /usr/lib/avr/include/avr/iomx8.h
#include <avr/interrupt.h>         // avr interrupt helpers - ISR, sei, cli
#include "./include/types.h"      // BOOL, BYTE, WORD, DWORD, FLOAT
#include "./include/system.h"     // standard startup
#include "./include/atomic.h"     // EI DI
#include "./include/led.h"        // status led
#include "./include/usart.h"      // serial i/o via usart
#include "./include/stdout.h"     // stdout via usart
#include "./include/reset.h"      // processor reset
#include "./include/bootloader.h" // LOADER_REQUEST_REBOOT
#include "./include/ticker.h"     // timer tick interrupt
#include "./include/twi.h"        // two-wire interface
#include "./include/delay.h"      // delay_ms

#include <math.h>                          // trig
#define DEG(X) ((X) * 57.2957795130823229) // radians to degrees
#define RAD(X) ((X) *  0.0174532925199433) // degrees to radians

typedef DWORD TICKS;               // an interval measured by AVR timer interrupt (spans 2^32 ticks = 50 days @ 1000Hz)

#include "version.h"               // date of issue
#include "config.h"                // configuration constants
#include "stack.h"                 // stack checker
#include "time.h"                  // timing functions
#include "lcd.h"                   // lcd display
#include "bat.h"                   // battery monitor
#include "pwr.h"                   // power controller
#include "servo.h"                 // servo controller
#include "gyro.h"                  // gyros
#include "acc.h"                   // accelerometers
#include "imu.h"                   // orientation tracker
#include "cam.h"                   // camera controller
#include "watchdog.h"              // watchdog timer
#include "timebase.h"              // time base generator
#include "err.h"                   // error handlers

// Foreground thread - main loop.
//
void
run()
   {
   // indicate that we're running
   //
   LED_on();
   
   // align imu, bike, and gyros with each other
   //
   LCD_clear(); LCD_printf(0, ("Calibrating..."));
   IMU_align();

   // enable camera motion
   //
   CAM_enabled = 1;
   
   // not much going on here, all the interesting action happens in interrupt-driven background thread
   //
   BOOL  timeit  = 0;          // debug
   TICKS tc_time = TIME_now(); // temperature compensation timer
   FLOAT maxLean = 0;          // highest lean angle seen so far
   FLOAT maxGees = 0;          // highest gee's seen so far
   for (;;)
      {
      while (!USART_ready())
         {
         // apply temperature compensation to gyro zero rate biases every few seconds
         //
         if (TIME_periodic(&tc_time, 5))
            GYRO_tc();
         
         // monitor battery voltage
         //
         if (BAT_low())
            ERR_battery();
            
         // update LCD display
         //
         FLOAT gees = -ACC_getX();
         FLOAT lean =  DEG(IMU_getRollAngle());

         if (fabs(gees) > fabs(maxGees)) maxGees = gees;
         if (fabs(lean) > fabs(maxLean)) maxLean = lean;

         LCD_printf(0, ("%+4.1fG %+3.0f %+5.2fG",  maxGees, maxLean, gees));
         LCD_bar(lean, 45);
         
         // time this loop to see how much cpu remains unused after interrupts take their share
         //
         if (timeit) 
            {
            static TICKS start;
            static WORD  n;
            if (++n == 100)
               {
               TICKS now   = TIME_now();
               TICKS ticks = now - start;
               printf("%.2f loops/sec\n", n / ((FLOAT)ticks / TICKER_HZ));
               n = 0;
               start = now;
               }
            }
         }
   
      switch (USART_get())
         {
         case 't': timeit = !timeit; break;
         case LOADER_REQUEST_REBOOT: reset_processor(); break;
         case 'q': printf("\n"); return;
         default:  printf("?\n");
         }
      }
   }

#include "workbench.h"

int
main(int argc, char **argv)
   {
   // discover reason for system reset - fetch status from bootloader (if present) or directly (if standalone)
   //
   BYTE mcusr = (argc == BOOTLOADER_MAGIC) ? (WORD)argv : MCUSR;
   MCUSR = 0;

   // initialize system basics
   //
   STACK_init();
   SYSTEM_init();
   USART_init();
   STDOUT_init();
   LED_init();

   // indicate that we're running
   //
   LED_on();
   printf("%s\n", Version);
   
   // initialize things that don't need watchdog timeouts
   //
   PWR_init();
   LCD_init();
   CONFIG_init();
   BAT_init();
   SERVO_init();
   
   // check for faults
   //
#if 0
   printf("\n[%02x", mcusr);
   if (mcusr & (1 << PORF))  printf(" power-on-reset");
   if (mcusr & (1 << BORF))  printf(" brownout-reset");
   if (mcusr & (1 << WDRF))  printf(" watchdog-reset");
   if (mcusr & (1 << EXTRF)) printf(" external-reset");
   printf("]\n");
#endif
   if      (mcusr & (1 << PORF)) ;              // poweron  reset
   else if (mcusr & (1 << BORF)) ERR_power();   // brownout reset
   else if (mcusr & (1 << WDRF)) ERR_timeout(); // watchdog reset
  
   // initialize sensors, using watchdog timeouts to guard TWI accesses (which can fail to complete)
   //
   WATCHDOG_start();
   TWI_init(ERR_twi);
   GYRO_init();
   ACC_init();
   
   // start background thread to run timebase, sensors, and imu
   //
   TIMEBASE_init();

   // check battery condition
   //
   if (BAT_low())
      ERR_battery();

#if 0
   // generate warmup data for gnuplot
   //
   while (!USART_ready()) 
      printf("%.2f %.1f %.1f %.1f\n", (FLOAT)TIME_now() / TICKER_HZ, DEG(GYRO_getRollRate()), DEG(GYRO_getPitchRate()), DEG(GYRO_getYawRate()));
   USART_get();
#endif

   // wait for gyros to stabilize
   //
   #define WARMUP 4.0 // warmup time, in seconds (choose based on plot of warmup data generated above)
   TICKS start = TIME_now();
   while (TIME_elapsed(&start) < WARMUP)
      {
      LCD_init(); LCD_printf(0, ("Warming up...")); LCD_printf(1, ("%.2fV", BAT_read()));
      TIME_pause(.5);
      }

   // start foreground thread to monitor everything
   //
   run();

   // debug
   //
   workbench();

   return 0;
   }
