// Controller for PWM driven servo.
//
// Units:      TIMER1
// Interrupts: none
// Pins:       PORTB1
// Clock:      8Mhz

// --------------------------------------------------------------------
//                        Implementation.
// --------------------------------------------------------------------

typedef SWORD SERVO_TENTHS; // a servo angle, in tenths of a degree

// Servo characteristics.
//
#define SERVO_CENTER_COUNTS     750 // pwm counts for 0 degrees of servo rotation
#define SERVO_COUNTS_PER_DEGREE 5   // pwm counts per degree of servo rotation (5=standard, 3=servocity 180 degree mod)
#define SERVO_LIMIT_TENTHS      900 // +/- travel limit, in tenths of a degree (HS-425BB servo)

// Convert servo angle, in tenths of a degree, to PWM counter value.
//
static WORD
SERVO_degreesToCounts(SERVO_TENTHS deg)
   {
   return SERVO_CENTER_COUNTS + (deg * SERVO_COUNTS_PER_DEGREE) / 10;
   }

// Convert PWM counter value to servo angle, in tenths of a degree.
//
static SERVO_TENTHS
SERVO_countsToDegrees(WORD cnt)
   {
   return (((SWORD)cnt - SERVO_CENTER_COUNTS) * 10) / SERVO_COUNTS_PER_DEGREE;
   }

// --------------------------------------------------------------------
//                          Interface.
// --------------------------------------------------------------------

// Prepare servo interface for use.
//
static void
SERVO_init()
   {
   // Set TIMER1 to generate 20ms period and nominal 1.5ms pulse width.
   //
   // We use waveform generation mode 8
   // "PWM, phase and frequency correct, TOP=ICR1".
   //
   // The counter, TCNT1, will run from 0 to TOP back down to 0.
   // We drive the OC1A pin "hi" during the portion of time
   // for which TCNT1 <= OCR1A.
   //
   // We use timer clock == system clock divided by 8.
   // For an 8MHz system clock and a TOP value of 10,000
   // this will yield a 50Hz (20ms) waveform:
   //
   //  8e6 / 8 / (1e4 upcounts + 1e4 downcounts) = .5e2 = 50Hz
   //
   //                     . _ _ _ _ _ _ _ _ _ TCNT1 == TOP 
   //                   .   .               .       == ICR1 = 10,000
   //                 .       .           .
   //       .       .           .       .
   //         .   . _ _ _ _ _ _ _ . _ . _ _ _ TCNT1 == OCR1A
   // TCNT1:    .                   . _ _ _ _ TCNT1 == BOTTOM = 0
   //
   //         . . .               . . .
   //         .   .               .   .
   // OC1A: . .   . . . . . . . . .   . . . .
   //                             |-+-|       2 x 750 counts = 1.5ms
   //           |---------+---------|         2 x 10,000 counts = 20ms
   
   TCCR1A =  
          0
                          // table 15-3
          | (0 << COM1A0) // clear OC1A on match when upcounting...
          | (1 << COM1A1) // ...set on match when downcounting

                          // table 15-4
          | (0 << WGM10)  // waveform generation mode 8
          | (0 << WGM11)  // waveform generation mode 8
          ;
   
   TCCR1B =
          0
                          // table 15-4
          | (0 << WGM12)  // waveform generation mode 8
          | (1 << WGM13)  // waveform generation mode 8

                          // table 15-5
          | (0 << CS10)   // timer clock = system clock / 8
          | (1 << CS11)   // timer clock = system clock / 8
          | (0 << CS12)   // timer clock = system clock / 8
          ;
   
#if CLOCK_MHZ != 8
   #error CLOCK_MHZ
#endif

   // TOP value for waveform width of 20ms at 8Mhz/8
   //
   ICR1 = 10000;
   
   // set center position
   //
   OCR1A = SERVO_CENTER_COUNTS;
   TCNT1 = 0;
   
   // begin outputting servo signal (configure PORTB1 as output for use by OC1A)
   //
   DDRB |= (1 << DDB1);
   }

// Get servo's current position angle, in tenths of a degree.
//
static SERVO_TENTHS
SERVO_get()
   {
   SERVO_TENTHS angle = SERVO_countsToDegrees(OCR1A);
   return angle;
   }

#define SERVO_FILTER_K Config.servo_filter_k

// Set servo's angular position.
// Taken: desired position angle, in tenths of a degree
//
void
SERVO_set(SERVO_TENTHS angle)
   {
   // don't exceed servo hard stops
   if      (angle < -SERVO_LIMIT_TENTHS) angle = -SERVO_LIMIT_TENTHS;
   else if (angle > +SERVO_LIMIT_TENTHS) angle = +SERVO_LIMIT_TENTHS;

   // smooth with a low pass filter
   // K = filter strength (0=none, 1=weak, 4+=strong)
   //
   const BYTE K = SERVO_FILTER_K;
   static SDWORD filter;
   filter = filter - (filter >> K) + angle;
   angle = filter >> K;
   
   OCR1A = SERVO_degreesToCounts(angle);
 //printf("\nservo %5d->%6d", angle, OCR1A);
   }
