// Hardware testing and calibration functions.
//

// Adjust battery monitor to give true volts.
//
void
adjust_battery()
   {
   for (;;)
      {
      while (!USART_ready())
         {
         WORD  adc   = ADC_read_battery();
         FLOAT volts = ADC_battery_counts_to_volts(adc);
         FLOAT pct   = (volts - 7.2) / (8.4 - 7.2) * 100; // 2s lipo is 7.2V to 8.4V (curve is not really linear, this is just an approximation)
         printf("0x%03x %4u %4.2fV %3.0f%% m=%5.5f\r", adc, adc, volts, pct, Config.battery_multiplier);
         }
      switch (USART_get())
         {
         case '=': Config.battery_multiplier += .00001; break; // raise indicated voltage
         case '-': Config.battery_multiplier -= .00001; break; // lower indicated voltage

         case 'q': printf("\n"); return;
         default:  printf("\n?\n");
         }
      }
   }
         
// Observe servo output.
//
void
adjust_servo()
   {
   BYTE saveK = SERVO_FILTER_K;
   SERVO_FILTER_K = 0;
   for (;;)
      {
      printf("%6d\r", SERVO_get());
      switch (USART_get())
         {
         case 'j': SERVO_set(SERVO_get() -  1 * 10); break;
         case 'k': SERVO_set(SERVO_get() +  1 * 10); break;
         case 'J': SERVO_set(-90 * 10);              break;
         case 'K': SERVO_set(+90 * 10);              break;
         case '.': SERVO_set(0); break;

         case 'q': printf("\n"); SERVO_FILTER_K = saveK; return;
         default:  printf("\n?\n");
         }
      }
   }

// Adjust accelerometer biases and gains to give true gees.
//
void
adjust_acc()
   {
#if HAVE_ACCELEROMETERS
   SWORD xmin = 9999, xmax = -9999,
         ymin = 9999, ymax = -9999,
         zmin = 9999, zmax = -9999;
           
   TICKS start = TIME_now();
   for (;;)
      {
      while (!USART_ready())
         printf("t=%6.2f x=%+6.3f y=%+6.3f z=%+6.3f r=%+6.1f p=%+6.1f k=%u\r", TIME_elapsed(&start), ACC_getX(), ACC_getY(), ACC_getZ(), DEG(ACC_getRollAngle()), DEG(ACC_getPitchAngle()), ACC_FILTER_K);
      
      DI();
      SWORD x = ACC_x;
      SWORD y = ACC_y;
      SWORD z = ACC_z;
      EI();

      switch (USART_get())
         {
         // set accelerometer smoothing strength
         //
         case 'A': if (ACC_FILTER_K != 8) ACC_FILTER_K += 1; break;
         case 'a': if (ACC_FILTER_K != 0) ACC_FILTER_K -= 1; break;

         // begin calibration by resetting biases and gains...
         //
         case 'r':
         ACC_X_BIAS = ACC_Y_BIAS = ACC_Z_BIAS = 0;
         ACC_X_GAIN = ACC_Y_GAIN = ACC_Z_GAIN = 0.001;
         break;
         
         // then orient each axis and hit xyzXYZ...
         //
         case 'x': xmin = x; break;
         case 'X': xmax = x; break;
         
         case 'y': ymin = y; break;
         case 'Y': ymax = y; break;
         
         case 'z': zmin = z; break;
         case 'Z': zmax = z; break;

         // finally hit c...
         //
         case 'c': ACC_X_BIAS =      ((xmax + xmin) / 2);
                   ACC_Y_BIAS =      ((ymax + ymin) / 2);
                   ACC_Z_BIAS =      ((zmax + zmin) / 2);

                   ACC_X_GAIN = 1. / ((xmax - xmin) / 2);
                   ACC_Y_GAIN = 1. / ((ymax - ymin) / 2);
                   ACC_Z_GAIN = 1. / ((zmax - zmin) / 2);
                   break;

         case '\r':printf("\n%.2fs\n", TIME_elapsed(&start)); start = TIME_now(); break; // stopwatch to measure accelerometer settling time
         case 'q': printf("\n"); return;
         default:  printf("\n?\n");
         }
      }
#else
   printf("no accelerometers\n");
#endif
   }

// Adjust gyro gains so angles derived from gyros match those derived from accelerometers.
//
void
adjust_gyro()
   {
   BYTE mode = 'A';
   
   #define XNEG Config.gyro_gain_xneg // roll
   #define XPOS Config.gyro_gain_xpos
   #define YNEG Config.gyro_gain_yneg // pitch
   #define YPOS Config.gyro_gain_ypos
   #define ZNEG Config.gyro_gain_zneg // yaw
   #define ZPOS Config.gyro_gain_zpos
   
   // turn off drift correction so imu will return pure gyro angles with no accelerometer feedback applied
   IMU_drift_correction_disabled = 1;
   
   for (;;)
      {
      while (!USART_ready())
         {
         FLOAT g_roll  = DEG(IMU_getRollAngle()),  a_roll  = DEG(ACC_getRollAngle()),
               g_pitch = DEG(IMU_getPitchAngle()), a_pitch = DEG(ACC_getPitchAngle()),
               g_yaw   = DEG(IMU_getYawAngle());

         switch (mode)
            {
            // rates about all axes
            case 'a': printf("%+6.1f:R %+6.1f:P %+6.1f:Y\r", DEG(GYRO_getRollRate()), DEG(GYRO_getPitchRate()), DEG(GYRO_getYawRate())); break;
                                
            // angles about all axes (gyro vs accelerometer)
            case 'A': printf("%+6.1f<g R a>%+-6.1f %+6.1f<g P a>%+-6.1f %+6.1f<g Y a>_\r", g_roll, a_roll, g_pitch, a_pitch, g_yaw); break;
            
            // angles about each axis (adjust gains)
            case 'r': printf("%9.7f %+6.1f<g r a>%+-6.1f(%+-4.1f)\r", XNEG, g_roll, a_roll, g_roll - a_roll); break;
            case 'R': printf("%9.7f %+6.1f<g R a>%+-6.1f(%+-4.1f)\r", XPOS, g_roll, a_roll, g_roll - a_roll); break;
            
            case 'p': printf("%9.7f %+6.1f<g p a>%+-6.1f(%+-4.1f)\r", YNEG, g_pitch, a_pitch, g_pitch - a_pitch); break;
            case 'P': printf("%9.7f %+6.1f<g P a>%+-6.1f(%+-4.1f)\r", YPOS, g_pitch, a_pitch, g_pitch - a_pitch); break;
            
            case 'y': printf("%9.7f %+6.1f<g y a>_\r", ZNEG, g_yaw); break;
            case 'Y': printf("%9.7f %+6.1f<g Y a>_\r", ZPOS, g_yaw); break;
            }
         }
      
      BYTE ch = USART_get();
      #define INC .0000001
      switch (ch)
         {
         case '=':
         switch (mode)
            {
            case 'r': XNEG  += INC;   break;
            case 'R': XPOS  += INC;   break;
            case 'p': YNEG  += INC;   break;
            case 'P': YPOS  += INC;   break;
            case 'y': ZNEG  += INC;   break;
            case 'Y': ZPOS  += INC;   break;
            }
         continue;
         
         case '-':
         switch (mode)
            {
            case 'r': XNEG  -= INC;   break;
            case 'R': XPOS  -= INC;   break;
            case 'p': YNEG  -= INC;   break;
            case 'P': YPOS  -= INC;   break;
            case 'y': ZNEG  -= INC;   break;
            case 'Y': ZPOS  -= INC;   break;
            }
         continue;
      #undef INC
         
         case 'n': // nominal (datasheet) gains on all axes
            XNEG =
            XPOS =
            YNEG =
            YPOS =
            ZNEG =
            ZPOS = .000156; // .00875 deg/sec/digit == ~.000156 radians/sec/digit
         continue;
         }

      printf("\n");
      switch (ch)
         {
         case 'A':
         case 'a':
         case 'r':
         case 'R':
         case 'p':
         case 'P':
         case 'y':
         case 'Y': mode = ch; break;
         
         case ' ': IMU_set(ACC_getRollAngle(), ACC_getPitchAngle(), 0); break;
         case '.': IMU_align(); break;
         case 'q': IMU_drift_correction_disabled = 0; return;
         default:  printf("?\n");
         }
      }
   }

// Observe and adjust IMU sensors and drift correction behavior.
//
void
adjust_imu()
   {
   BOOL  saveCam = CAM_enabled; CAM_enabled = 0;
   TICKS tc_time = TIME_now();
   
   #ifndef IMU_RATE_THRESHOLD
   static FLOAT zero;
   #define IMU_RATE_THRESHOLD zero
   #endif

   for (;;)
      {
      // output current "tuning" parameters, emitting a leading newline so gnuplot will generate a line break on graph
      //
      printf("\n#Kg=%u Ka=%u rat=%.1f dur=%.3f tim=%.1f dft=%c cam=%c\n", GYRO_FILTER_K, ACC_FILTER_K, DEG(IMU_RATE_THRESHOLD), IMU_RATE_DURATION, IMU_TIME_CONSTANT, IMU_drift_correction_disabled ? '-' : '+', CAM_enabled ? '+' : '-');
         
      // output angles and errors for gnuplot
      //
      while (!USART_ready())
         {
         // apply temperature correction
         //
         if (TIME_periodic(&tc_time, 5))
            GYRO_tc();
      
         if (!IMU_debug_snapshots_enabled) continue;

         // output data
         //
         DI();
         FLOAT t = (FLOAT)TIME_now() / TICKER_HZ;
         SBYTE T = GYRO_temperature();
         FLOAT i_roll  = IMU_getRollAngle(),  a_roll  = ACC_getRollAngle(),  e_roll  = IMU_rollError,  c_roll  = IMU_rollCorr;
         FLOAT i_pitch = IMU_getPitchAngle(), a_pitch = ACC_getPitchAngle(), e_pitch = IMU_pitchError, c_pitch = IMU_pitchCorr;
         FLOAT i_yaw   = IMU_getYawAngle(),   a_yaw   = 0,                   e_yaw   = IMU_yawError,   c_yaw   = IMU_yawCorr;
         EI();
         
         //                -----------roll---------------- ----------pitch---------------- -----------yaw-----------------
         //      time  temp    acco   gyro   erro   corr       acco   gyro   erro   corr       acco   gyro   erro   corr
         printf("%6.2f %2d { %+8.3f %+8.3f %+8.3f %+8.3f } { %+8.3f %+8.3f %+8.3f %+8.3f } { %+8.3f %+8.3f %+8.3f %+8.3f }\n",
            t, T, 
            DEG(a_roll),  DEG(i_roll),  DEG(e_roll),  DEG(c_roll),
            DEG(a_pitch), DEG(i_pitch), DEG(e_pitch), DEG(c_pitch),
            DEG(a_yaw),   DEG(i_yaw),   DEG(e_yaw),   DEG(c_yaw)
            );
         }

      switch (USART_get())
         {
         // toggle data output
         //
         case ' ': IMU_debug_snapshots_enabled = !IMU_debug_snapshots_enabled; break;
         
         // toggle camera tracking
         //
         case 'c': CAM_enabled = !CAM_enabled; break;
         
         // toggle drift correction
         //
         case '*': IMU_drift_correction_disabled = !IMU_drift_correction_disabled; break;
         
         // set gyro smoothing strength
         //
         case 'G': if (GYRO_FILTER_K != 8) GYRO_FILTER_K += 1; break;
         case 'g': if (GYRO_FILTER_K != 0) GYRO_FILTER_K -= 1; break;

         // set accelerometer smoothing strength
         //
         case 'A': if (ACC_FILTER_K != 8) ACC_FILTER_K += 1; break;
         case 'a': if (ACC_FILTER_K != 0) ACC_FILTER_K -= 1; break;

         // set drift correction turn rate threshold
         //
         #define INC RAD(.1)
         case 'R': IMU_RATE_THRESHOLD += INC; break;
         case 'r': IMU_RATE_THRESHOLD -= INC; if (IMU_RATE_THRESHOLD < 0) IMU_RATE_THRESHOLD = 0;  break;
         #undef INC
         
         // set drift correction turn rate duration
         //
         #define INC .005
         case 'D': IMU_RATE_DURATION += INC; break;
         case 'd': IMU_RATE_DURATION -= INC; if (IMU_RATE_DURATION < 0) IMU_RATE_DURATION = 0;  break;
         #undef INC
         
         // set drift correction time constant
         //
         #define INC .1
         case 'T': IMU_TIME_CONSTANT += INC; break;
         case 't': IMU_TIME_CONSTANT -= INC; if (IMU_TIME_CONSTANT < 0) IMU_TIME_CONSTANT = 0;  break;
         #undef INC
         
         // observe sensor activity
         // - time should match wall clock
         // - gyro hits should progress at GYRO_HZ
         // - acc  hits should progress at ACC_HZ
         // - imu  hits should progress at IMU_HZ
         // - cam  hits should progress at CAM_HZ
         // - misses should be zero
         //
         case '?':
         {
         DI();
         ACC_hits = GYRO_hits = IMU_hits = CAM_hits = 0;
         TICKS start = TIME_now();
         EI();
         while (!USART_ready())
            {
            FLOAT t = TIME_elapsed(&start);
            printf("\rt=%-6.2f a=%.1f/%1u g=%-5.1f/%1u i=%-5.1f c=%.1f", t, ACC_hits / t, ACC_misses, GYRO_hits / t, GYRO_misses, IMU_hits / t, CAM_hits / t);
            }
         USART_get();
         }
         break;
         
         case '.': IMU_align(); break;
         case 'q': CAM_enabled = saveCam; IMU_debug_snapshots_enabled = 0; IMU_drift_correction_disabled = 0; return;
         default:  printf("?\n");
         }
      }
   }

// Adjust camera centering and travel gains.
//
void
adjust_camera()
   {
   // turn off drift correction so camera won't auto-recenter
   IMU_drift_correction_disabled = 1;

   CAM_enabled = 1;
   BOOL joggle = 0;
   for (;;)
      {
      FLOAT roll  = 0;
      FLOAT pitch = 0;
      
      while (!USART_ready())
         {
         roll  = DEG(IMU_getRollAngle());
         pitch = DEG(IMU_getPitchAngle());
         printf("C=%+4.1f L=%.1f R=%+-4.1f r=%+-6.1f p=%+-6.1f Ks=%u\r", CAM_BIAS, CAM_L_GAIN, CAM_R_GAIN, roll, pitch, SERVO_FILTER_K);
         }
         
      BYTE mode;
      if      (roll < -20) mode = 'L';
      else if (roll > +20) mode = 'R';
      else                 mode = 'C';

      
      BYTE ch = USART_get();
      switch (ch)
         {
         case '=':
         switch (mode)
            {
            case 'C': CAM_BIAS   += .1; break;
            case 'L': CAM_L_GAIN += .1; break;
            case 'R': CAM_R_GAIN += .1; break;
            }
         continue;
         
         case '-':
         switch (mode)
            {
            case 'C': CAM_BIAS   -= .1; break;
            case 'L': CAM_L_GAIN -= .1; break;
            case 'R': CAM_R_GAIN -= .1; break;
            }
         continue;
         }
         
      printf("\n");
      switch (ch)
         {
         case 'S': if (SERVO_FILTER_K != 8) SERVO_FILTER_K += 1; break;
         case 's': if (SERVO_FILTER_K != 0) SERVO_FILTER_K -= 1; break;
         
         case ' ': // joggle servo to remove stiction: turn and recenter, first from one direction then the other
                   if ((joggle = !joggle)) IMU_set(RAD(-5), 0, 0);
                   else                    IMU_set(RAD(+5), 0, 0);
                   TIME_pause(.5);
                   IMU_set(0,0,0);
                   break;

         
         case '.': IMU_align(); break;
         case 'q': IMU_drift_correction_disabled = 0; return;
         default:  printf("?\n");
         }
      }
   }

// Generate data for plotting gyro output vs temperature (units = gyro-digits).
//
void
plot_tc()
   {
   while (!USART_ready())
      {
      GYRO_x_sum = GYRO_y_sum = GYRO_z_sum = GYRO_cnt = 0;
      GYRO_calibrating = 1;
      TIME_pause(1);
      GYRO_calibrating = 0;
      printf("%d %ld %ld %ld %d\n", GYRO_temperature(), GYRO_x_sum / GYRO_cnt, GYRO_y_sum / GYRO_cnt, GYRO_z_sum / GYRO_cnt, GYRO_cnt);
      TIME_pause(5);
      }
   USART_get();
   }

// Generate data for plotting gyro output vs bandwidth (units = deg/sec).
//
void
plot_bw()
   {
   BYTE saveK = GYRO_FILTER_K;
   GYRO_FILTER_K = 0;

   GYRO_calibrate();
   while (!USART_ready())
      {
      printf("%.2f %.1f %.1f %.1f\n", (FLOAT)TIME_now() / TICKER_HZ, DEG(GYRO_getRollRate()), DEG(GYRO_getPitchRate()), DEG(GYRO_getYawRate()));

      static BYTE  step;
      static TICKS time;
      if (TIME_periodic(&time, 10))
         switch (step++)
            {
            case 4: step = 0; // fall through
            case 0: { printf("# 12.5Hz\n"); DI(); TWI_write(GYRO_ADDR, GYRO_CTRL_REG1, 0x4F); EI(); break; }
            case 1: { printf("# 25Hz\n");   DI(); TWI_write(GYRO_ADDR, GYRO_CTRL_REG1, 0x5F); EI(); break; }
            case 2: { printf("# 50Hz\n");   DI(); TWI_write(GYRO_ADDR, GYRO_CTRL_REG1, 0x6F); EI(); break; }
            case 3: { printf("# 70Hz\n");   DI(); TWI_write(GYRO_ADDR, GYRO_CTRL_REG1, 0x7F); EI(); break; }
            }
      }
   USART_get();
   
   DI(); GYRO_init(); EI();
   GYRO_FILTER_K = saveK;
   }
      
// Generate data for plotting gyro output vs engine vibration (units = deg/sec).
//
void
plot_vib()
   {
   while (!USART_ready())
      printf("%.2f %.1f %.1f %.1f %.1f\n", (FLOAT)TIME_now() / TICKER_HZ, DEG(GYRO_getRollRate()), DEG(GYRO_getPitchRate()), DEG(GYRO_getYawRate()), IMU_upright ? DEG(IMU_RATE_THRESHOLD) : 0);
   USART_get();
   }

#if 0
// Timing tests.
// Note: bypass function calls in TICKER_interrupt() before running these tests.
//
volatile FLOAT Dummy;

void
dummy()
   {
   GYRO_update();
   ACC_update();
   IMU_update();
   CAM_update();
// Dummy = IMU_getYawAngle();
// Dummy = IMU_getPitchAngle();
   }

void
timing_tests()
   {
   TICKS start = TIME_now();
   WORD  updates = 0;
   while (!USART_ready())
      {
      //                         old software,    new software,      cam2
      //                         TICKER_HZ=200,   TICKER_HZ=400,    6Mar2012
      //                         TWI=50kHz         TWI=100kHz
      //
      // GYRO_update              1.2 ms              .7 ms            .5 ms
      // ACC_update               1.1 ms              .6 ms           n/a
      // IMU_update               3.6 ms             1.7 ms           1.5 ms
      // CAM_update               .75 ms              .5 ms            .3 ms
      // combined                  n/a               n/a              3.9 ms <- not sure why sums don't add up. inlining/optimization effects?
      // IMU_getYawAngle          .65 ms              .4 ms            .2 ms
      // IMU_getPitchAngle        .35 ms              .3 ms            .1 ms

      dummy();
      if (++updates == 500)
         {
         FLOAT seconds = (FLOAT)(TIME_now() - start) / TICKER_HZ;
         printf("t=%.2f, updates/sec=%.0f, %.1f ms/update\n", seconds, updates / seconds, 1000.0 * seconds / updates);
         start = TIME_now();
         updates = 0;
         }
      }
   }
#else
void timing_tests() {}
#endif

// Run test and configuration functions.
//
void
workbench()
   {
 //timing_tests();
   for (;;)
      {
      LED_on();
      CAM_enabled = 0;
      printf("%u I)nit D)ump b)attery s)ervo a)cc g)yro i)mu c)amera t)c B)w v)ib tw)i tim)eout p)ower r)un", STACK_free());

      char ch = USART_get();
      printf("\n");
      
      switch (ch)
         {
         case 'I': CONFIG_init();                       break;
         case 'D': CONFIG_dump();                       break;
         
         case 'b': adjust_battery();                    break;
         case 's': adjust_servo();                      break;
         case 'a': adjust_acc();                        break;
         case 'g': adjust_gyro();                       break;
         case 'i': adjust_imu();                        break;
         case 'c': adjust_camera();                     break;
         
         case 't': plot_tc();                           break;
         case 'B': plot_bw();                           break;
         case 'v': plot_vib();                          break;
         
         case 'w': TWI_write(0x77, 7, 7);               break; // force a TWI error by attempting to access a non-existent device
         case 'm': { DI(); for (;;); EI(); }            break; // force a watchdog timeout error
         case 'p': PWR_off(); continue;                 break; // turn off the power
         
         case 'r': run();                               break; // resume normal operation
         case LOADER_REQUEST_REBOOT: reset_processor(); break; // load new software
         default:  printf("?\n");                       break;
         }
      }
   }
